<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\Response;
use Sentry\SentrySdk;
use Sentry\Tracing\TransactionContext;

class SentryTracingMiddleware
{
    public function handle(Request $request, Closure $next): Response
    {
        if (!class_exists(\Sentry\SentrySdk::class)) {
            return $next($request); // Sentry not available
        }

        // Set user info (Crash-Free Users)
        \Sentry\configureScope(function (\Sentry\State\Scope $scope) {
            if (auth()->check()) {
                $user = auth()->user();
                $scope->setUser([
                    'id' => $user->id,
                    'email' => $user->email ?? null,
                    'username' => $user->name ?? null,
                ]);
            } else {
                $scope->setUser([
                    'id' => 'guest',
                    'username' => 'Guest User',
                ]);
            }
        });

        // Start Sentry Performance Transaction
        $context = new TransactionContext();
        $context->setName($request->method() . ' ' . $request->path());
        $context->setOp('http.server');

        $transaction = \Sentry\startTransaction($context);
        SentrySdk::getCurrentHub()->setSpan($transaction);

        try {
            $response = $next($request);
        } catch (\Throwable $e) {
            \Sentry\captureException($e);
            throw $e;
        } finally {
            $transaction->finish();
        }

        return $response;
    }
}

