<?php

namespace App\Http\Controllers;

use App\Models\User;
use Illuminate\Http\Request;
use Spatie\Permission\Models\Role;
use Illuminate\Support\Facades\Hash;
use App\Models\Department;
use App\Models\District;
use App\Mail\UserCreatedMail;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Auth;
use App\Helpers\CommonHelper;

class UserController extends Controller
{
    public function __construct()
    {
        $this->middleware('permission:View User', ['only' => ['index']]);
        $this->middleware('permission:Create User', ['only' => ['create', 'store']]);
        $this->middleware('permission:Update User', ['only' => ['update', 'edit']]);
        $this->middleware('permission:Delete User', ['only' => ['destroy']]);
        $this->middleware('auth');
    }

    public function index()
    {
        $users = User::get();
        return view('admin_pages.access-control.user.index', ['users' => $users]);
    }

    public function create()
    {
        $roles = Role::pluck('name', 'name')->all();
        // BOC: Added code to get the departments and districts. BY ALMAAZ AHMED on 05-May-2025 PMS: 160193
        $departments = Department::where('status', 'Active')->get();
        $districts = District::where('status', 'Active')->get();
        return view('admin_pages.access-control.user.create', [
            'roles' => $roles,
            'departments' => $departments,
            'districts' => $districts
        ]);
        // EOC: Added code to get the departments and districts. BY ALMAAZ AHMED on 05-May-2025 PMS: 160193
    }

    /**
     * BOC: Added function to decode the password from the encoded form
     * @param $encoded
     * @return string
     * 
     * Author: ALMAAZ AHMED
     * Date: 21-March-2025
     * PMS: 148137
     */
    // Function to decode the password from the encoded form
    private function decodePassword($encoded)
    {
        // Decode the first base64 encoding
        $encoded = base64_decode($encoded);

        $decoded = "";
        for ($i = 0; $i < strlen($encoded); $i++) {
            // XOR the characters with 10 to reverse the encryption
            $a = ord($encoded[$i]);
            $b = $a ^ 10; // XOR with the same number (10) used for encoding
            $decoded .= chr($b); // Reconstructed decoded string
        }

        // Decode the second base64 encoding to get the original password
        return base64_decode(base64_decode($decoded));
    }
    // EOC: Added function to decode the password from the encoded form

    public function store(Request $request)
    {
        // BOC: Added code to decode the password from the encoded form. BY ALMAAZ AHMED on 21-March-2025 PMS: 148137
        $passwordEncoded = $request->password;
        // Decode the encrypted passwords
        $password = $this->decodePassword($passwordEncoded);

        // Merge decoded passwords back into the request
        $request->merge(['password' => $password]);
        // EOC: Added code to decode the password from the encoded form. BY ALMAAZ AHMED on 21-March-2025 PMS: 148137
        
        // BOC: Added code to validate the request. BY ALMAAZ AHMED on 05-May-2025 PMS: 160193
        $request->validate([
            'username' => 'required|string|max:255|unique:users,email',
            'email_address' => 'required|email|max:255|unique:users,email_address',
            'password' => 'required|string|min:5|max:20',
            'role' => 'required',
            'department_id' => 'required_if:role,Department-User,Department-Approver,Department-HOD',
            'district_id' => 'required_if:role,District-User,District-Approver,District-HOD'
        ]);

        $user = User::create([
            'email' => $request->username,
            'email_address' => $request->email_address,
            'password' => Hash::make($request->password),
            'department_id' => $request->department_id ?: null,
            'district_id' => $request->district_id ?: null,
        ]); 
        // EOC: Added code to validate the request. BY ALMAAZ AHMED on 05-May-2025 PMS: 160193

        $user->syncRoles($request->role);

        Mail::to($user->email_address)->queue(new UserCreatedMail($user,$password));

        return redirect('/users')->with('status', 'User created successfully with roles');
    }

    public function edit(User $user)
    {
        $roles = Role::pluck('name', 'name')->all();
        $userRoles = $user->roles->pluck('name', 'name')->all();
        // BOC: Added code to get the departments and districts. BY ALMAAZ AHMED on 05-May-2025 PMS: 160193
        $departments = Department::where('status', 'Active')->get();
        $districts = District::where('status', 'Active')->get();
        return view('admin_pages.access-control.user.edit', [
            'user' => $user,
            'roles' => $roles,
            'userRoles' => $userRoles,
            'departments' => $departments,
            'districts' => $districts
        ]);
        // EOC: Added code to get the departments and districts. BY ALMAAZ AHMED on 05-May-2025 PMS: 160193
    }

    public function update(Request $request, User $user)
    {
        // BOC: Added code to decode the password from the encoded form. BY ALMAAZ AHMED on 21-March-2025 PMS: 148137
        $passwordEncoded = $request->password;
        // Decode the encrypted passwords
        $password = $this->decodePassword($passwordEncoded);

        // Merge decoded passwords back into the request
        $request->merge(['password' => $password]);
        // EOC: Added code to decode the password from the encoded form. BY ALMAAZ AHMED on 21-March-2025 PMS: 148137

        // BOC: Added code to validate the request. BY ALMAAZ AHMED on 05-May-2025 PMS: 160193
        $request->validate([
            'username' => 'required|string|max:255|unique:users,email,' . $user->id,
            'email_address' => 'required|email|max:255|unique:users,email_address,' . $user->id,
            'password' => 'nullable|string|min:5|max:20',
            'role' => 'required',
            'department_id' => 'required_if:role,Department-User,Department-Approver,Department-HOD',
            'district_id' => 'required_if:role,District-User,District-Approver,District-HOD'
        ]);

        // Initialize data array
        $data = [
            'email' => $request->username,
            'email_address' => $request->email_address,
            // Check the role and clear district_id or department_id accordingly
            'department_id' => in_array($request->role, ['Department-User', 'Department-Approver', 'Department-HOD']) ? $request->department_id : null,
            'district_id' => in_array($request->role, ['District-User', 'District-Approver', 'District-HOD']) ? $request->district_id : null,
        ];

        // EOC: Added code to validate the request. BY ALMAAZ AHMED on 05-May-2025 PMS: 160193

        if (!empty($password)) {
            $data += [
                'password' => Hash::make($password),
            ];
        }

        // Update the user data
        $user->update($data);

        // Sync the roles with the updated role
        $user->syncRoles($request->role);

        // Redirect with success message
        return redirect('/users')->with('status', 'User Updated Successfully with roles');
    }

    public function destroy($userId)
    {
        $user = User::findOrFail($userId);
        $user->delete();

        return redirect('/users')->with('status', 'User Delete Successfully');
    }

    public function changePassword(){
        return view('admin_pages.access-control.user.change-password');
    }

    public function updatePassword(Request $request)
    {
        // Get the encrypted password and confirm password from the request
        $old_passwordEncoded = $request->old_password;
        $passwordEncoded = $request->password;
        $confirmPasswordEncoded = $request->password_confirmation;

        // Decode the encrypted passwords
        $old_password = $this->decodePassword($old_passwordEncoded);
        $password = $this->decodePassword($passwordEncoded);
        $confirmPassword = $this->decodePassword($confirmPasswordEncoded);

        // Merge decoded passwords back into the request
        $request->merge(['old_password' => $old_password, 'password' => $password, 'password_confirmation' => $confirmPassword]);

        // Validate the password and confirmation password
        $request->validate([
            'old_password' => 'required',
            'password' => 'required|confirmed|min:5|max:20',
        ], [
            'password.confirmed' => 'Password and Confirm Password should be the same',
        ]);

        try{
            $user = Auth::user();
        
            // Check if the old password matches the current password
            if (!Hash::check($request->old_password, $user->password)) {
                return back()->withErrors(['old_password' => 'Old Password Not Matched']);
            }
    
            // Hash and save the new password
            $user->password = Hash::make($request->password);
            $user->save();

            if ($user) {
                // Clear the last_logged_in field
                $user->last_logged_in = null;
                $user->save();
            }

            Auth::logout();
            $request->session()->invalidate();
            $request->session()->regenerateToken();
    
            // Log the user out from other devices
            // Auth::logoutOtherDevices($request->password);
    
            return redirect()->route('login')->withErrors(['email' => 'Password Changed successfully. Please login again.']);
        }catch(\Throwable $e){
            $logData['file'] = __FILE__;
            $logData['method'] = __FUNCTION__;
            $logData['exception'] = $e->getMessage();
            // Log exception to file with file name and method information
            CommonHelper::dataActionLogger($logData, 'error');
            return redirect('graph-dashboard')->with('error', 'Error while updating password. Please try after sometime.');
        }catch(QueryException $e){
            $logData['file'] = __FILE__;
            $logData['method'] = __FUNCTION__;
            $logData['exception'] = $e->getMessage();
            // Log exception to file with file name and method information
            CommonHelper::dataActionLogger($logData, 'error');
            return redirect('graph-dashboard')->with('error', 'Error while updating password. Please try after sometime.');
        }catch(\Exception $e){
            $logData['file'] = __FILE__;
            $logData['method'] = __FUNCTION__;
            $logData['exception'] = $e->getMessage();
            // Log exception to file with file name and method information
            CommonHelper::dataActionLogger($logData, 'error');
            return redirect('graph-dashboard')->with('error', 'Error while updating password. Please try after sometime.');
        }
    }

}
