<?php

namespace App\Http\Controllers;

use App\Helpers\PublicPageHelper;
use Illuminate\Http\Request;
use App\Models\Target;
use Maatwebsite\Excel\Facades\Excel;
use App\Exports\TargetExport;
use App\Helpers\CommonHelper;
use Illuminate\Database\QueryException;
use Illuminate\Support\Facades\Log;
use App\Http\Requests\TargetRequest;

class TargetController extends Controller
{
    public function __construct()
    {
        $this->middleware('permission:View Target', ['only' => ['index']]);
        $this->middleware('permission:Create Target', ['only' => ['create', 'store']]);
        $this->middleware('permission:Update Target', ['only' => ['update', 'edit']]);
        $this->middleware('permission:Delete Target', ['only' => ['destroy']]);
        $this->middleware('permission:Export Target', ['only' => ['exportTargets']]);
        $this->middleware('auth');
    }

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = Target::query();
        $query->join('goals', 'targets.goal_id', '=', 'goals.id');
        $query->select('targets.target_id', 'targets.target_name', 'targets.target_number', 'targets.status', 'goals.goal_name');
        $query->orderBy('targets.target_id', 'asc');
        if ($request->has('search')) {
            $query->where('goal_id', $request->search);
        }
        try {
            $targets = $query->paginate(env('ALL_PAGINATION_LIMIT', 10));
        } catch (QueryException $e) {
            $logData['file'] = __FILE__;
            $logData['method'] = __FUNCTION__;
            $logData['query'] = $e->getSql();
            $logData['exception'] = $e->getMessage();
            // Log exception to file with file name and method information
            CommonHelper::dataActionLogger($logData, 'exception');

            return redirect('targets')->with('error', 'Error while fetching Targets. Please try after sometime.');
        }
        return view('admin_pages.target.index', ['targets' => $targets]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $goals = CommonHelper::getGoalData('all');
        if(!$goals){
            $goals = [];
        }
        $target = [];
        return view('admin_pages.target.create', ['target' => $target, 'goals' => $goals]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(TargetRequest $request)
    {
        try {
            $target = new Target();
            $target->target_number = trim($request->target_number);
            $target->target_name = trim($request->target_name);
            $target->goal_id = trim($request->goal);
            $target->save();
        } catch (QueryException $e) {
            $logData['file'] = __FILE__;
            $logData['method'] = __FUNCTION__;
            $logData['query'] = $e->getSql();
            $logData['exception'] = $e->getMessage();
            // Log exception to file with file name and method information
            CommonHelper::dataActionLogger($logData, 'exception');

            return redirect('targets')->with('error', 'Error while creating Target. Please try after sometime.');
        }
        return redirect('targets')->with('success', 'Target created successfully.');
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $goals = CommonHelper::getGoalData('all');
        if(!$goals){
            $goals = [];
        }
        $query  = Target::query();
        try {
            $target = $query->where('target_id', $id)->first();
        } catch (QueryException $e) {
            $logData['file'] = __FILE__;
            $logData['method'] = __FUNCTION__;
            $logData['query'] = $e->getSql();
            $logData['exception'] = $e->getMessage();
            // Log exception to file with file name and method information
            CommonHelper::dataActionLogger($logData, 'exception');

            return redirect('targets')->with('error', 'Error while fetching the details to Edit. Please try after sometime.');
        }
        return view('admin_pages.target.create', ['target' => $target, 'goals' => $goals]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(TargetRequest $request, Target $target)
    {
        if (!empty($target->target_id)) {
            try {
                $target->target_number = trim($request->target_number);
                $target->target_name = trim($request->target_name);
                $target->goal_id = trim($request->goal);
                $target->save();
            } catch (QueryException $e) {
                $logData['file'] = __FILE__;
                $logData['method'] = __FUNCTION__;
                $logData['query'] = $e->getSql();
                $logData['exception'] = $e->getMessage();
                // Log exception to file with file name and method information
                CommonHelper::dataActionLogger($logData, 'exception');

                return redirect('targets')->with('error', 'Error while updating Target. Please try after sometime.');
            }
        } else {
            return redirect('targets')->with('error', 'Target not found.');
        }

        return redirect("targets")->with('success', 'Target updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id, string $action)
    {
        $query  = Target::query();
        try {
            $target = $query->find(trim($id));
            if (isset($target) && !empty($target)) {
                if (strtolower(trim($action)) == 'active') {
                    $target->status = 'Active';
                    $target->save();
                    return redirect('targets')->with('success', 'Target activated successfully.');
                } else if (strtolower(trim($action)) == 'inactive') {
                    $target->status = 'Inactive';
                    $target->save();
                    return redirect('targets')->with('success', 'Target deactivated successfully.');
                } else {
                    return redirect('targets')->with('error', 'Invalid action!');
                }
            } else {
                return redirect('targets')->with('error', 'Target not found!');
            }
        } catch (QueryException $e) {
            $logData['file'] = __FILE__;
            $logData['method'] = __FUNCTION__;
            $logData['query'] = $e->getSql();
            $logData['exception'] = $e->getMessage();
            // Log exception to file with file name and method information
            CommonHelper::dataActionLogger($logData, 'exception');

            return redirect('target')->with('error', 'Error while target status. Please try after sometime.');
        }
    }
    public function show($id)
    {
        // Logic for showing a specific target
    }

    public function exportTargets()
    {
        $query = Target::query();
        try {
            $query->join('goals', 'targets.goal_id', '=', 'goals.id');
        } catch (QueryException $e) {
            $logData['file'] = __FILE__;
            $logData['method'] = __FUNCTION__;
            $logData['query'] = $e->getSql();
            $logData['exception'] = $e->getMessage();
            // Log exception to file with file name and method information
            CommonHelper::dataActionLogger($logData, 'exception');
            return redirect('targets')->with('error', 'Error while exporting Targets. Please try after sometime.');
        }
        $targets = $query->select('targets.target_id', 'targets.target_number', 'targets.target_name', 'goals.goal_name', 'targets.status')->get();

        if ($targets->isEmpty()) {
            return redirect('targets')->with('error', 'No Targets found to export.');
        }

        $filename = 'list_of_targets_' . now()->format('Y-m-d_H-i-s') . '.xlsx';
        Excel::store(new TargetExport($targets), 'export/targets/' . $filename);

        // Generate the full path to the saved file
        $filePath = storage_path('app/export/targets/' . $filename);

        return response()->download($filePath, $filename);
    }
}
