<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Indicator;
use Maatwebsite\Excel\Facades\Excel;
use App\Exports\IndicatorExport;
use App\Helpers\CommonHelper;
use Illuminate\Support\Facades\DB;
use App\Models\IndicatorValueType;
use App\Models\ReportingYear;
use Illuminate\Database\QueryException;
use Illuminate\Support\Facades\Log;
use App\Models\Department;
use App\Models\Tag;
use App\Models\Target;
use App\Http\Requests\IndicatorRequest;
use App\Http\Requests\UpdateIndicatorRequest;
use App\Models\IndicatorDataEntry;
use App\Models\District;
use Doctrine\DBAL\Query;
use Error;

class IndicatorController extends Controller
{
    public function __construct()
    {
        $this->middleware('permission:View Indicator', ['only' => ['index']]);
        $this->middleware('permission:Create Indicator', ['only' => ['create', 'store']]);
        $this->middleware('permission:Update Indicator', ['only' => ['update', 'edit']]);
        $this->middleware('permission:Delete Indicator', ['only' => ['destroy']]);
        $this->middleware('permission:Export Indicator', ['only' => ['exportIndicators']]);
        $this->middleware('auth');
    }

    public function index(Request $request)
    {
        $districts = [];
        $departments = [];
        $targets = [];
        $indicators = [];

        $goals = CommonHelper::getGoalData('all');
        if (!$goals) {
            $goals = [];
        }
        if ($request->has('goal') && !empty($request->input('goal'))) {
            $goal = $request->input('goal');
            if (isset($goal) && !empty($goal)) {
                //check if $request->goal_id is integer value
                if (is_numeric($request->goal)) {
                    $targets = CommonHelper::getTargets('all', $request->goal);
                    if (!$targets) {
                        $targets = [];
                    }
                } else {
                    $targets = CommonHelper::getTargets('all');
                    if (!$targets) {
                        $targets = [];
                    }
                }
            } else {
                $targets = CommonHelper::getTargets('all');
                if (!$targets) {
                    $targets = [];
                }
            }
        } else {
            $targets = CommonHelper::getTargets('all', '1');
            if (!$targets) {
                $targets = [];
            }
        }

        $departments = CommonHelper::getDepartments('all');
        if (!$departments) {
            $departments = [];
        }

        $districts = CommonHelper::getDistricts('all');
        if (!$districts) {
            $districts = [];
        }

        $query = Indicator::query();
        $query->join('goals', 'indicators.goal_id', '=', 'goals.id');
        $query->join('targets', 'indicators.target_id', '=', 'targets.target_id');
        $query->select('indicators.id as indicator_id', 'indicators.indicator_number as indicator_number', 'indicators.indicator_name as indicator_name', 'goals.goal_name as goal_name', 'targets.target_name as target_name', 'indicators.unit as unit', 'indicators.is_positive_indicator as indicator_marker', 'indicators.is_discontinue as is_discontinue', 'indicators.baseline_year as baseline_year', 'indicators.baseline_year_id as baseline_year_id');
        $query->orderBy('indicators.id', 'desc');

        if ($request->has('goal') && !empty($request->input('goal'))) {
            //check if $request->goal_id is integer value
            if (is_numeric($request->goal)) {
                $query->where('indicators.goal_id', $request->goal);
                $default_goal = $request->goal;
            }
        } else {
            $query->where('indicators.goal_id', '1');
            $default_goal = 1;
        }
        if ($request->has('target') && !empty($request->input('target'))) {
            //check if $request->target_id is integer value
            if (is_numeric($request->target)) {
                $query->where('indicators.target_id', $request->input('target'));
            }
        }

        if ($request->has('type') && !empty($request->input('type'))) {
            if ($request->input('type') == 'dif') {
                $query->where('indicators.is_dif', 'Yes');
                $type = 'dif';
            } else if ($request->input('type') == 'utif') {
                $query->where('indicators.is_utif', 'Yes');
                $type = 'utif';
            }
        } else {
            $query->where('indicators.is_utif', 'Yes');
            $type = 'utif';
        }

        try {
            $indicators = $query->paginate(env('INDICATOR_PAGINATION_LIMIT', 30));
        } catch (QueryException $e) {
            $logData['file'] = __FILE__;
            $logData['method'] = __FUNCTION__;
            $logData['query'] = $e->getSql();
            $logData['exception'] = $e->getMessage();
            // Log exception to file with file name and method information
            CommonHelper::dataActionLogger($logData, 'exception');
            $indicators = [];
        }
        if (!empty($indicators)) {
            foreach ($indicators as $indicator) {
                $query = IndicatorDataEntry::query();
                $query->where('indicator_id', $indicator->indicator_id);
                $query->where('reporting_year_id', '!=', $indicator->baseline_year_id);
                $indicator_data_entries = $query->whereNotNull('indicator_value')->get('indicator_id');
                if (!empty($indicator_data_entries) && $indicator_data_entries->count() > 0) {
                    $indicator->is_editable = 'no';
                } else {
                    $indicator->is_editable = 'yes';
                }
            }
        }

        return view('admin_pages.indicator.index', ['indicators' => $indicators, 'targets' => $targets, 'districts' => $districts, 'departments' => $departments, 'goals' => $goals, 'default_goal' => $default_goal, 'type' => $type]);
    }

    private function getYearsRange()
    {
        // Get the minimum reporting year
        $minYear = ReportingYear::min('reporting_year');

        // Get the default reporting year
        $defaultYear = ReportingYear::where('is_default', 'Yes')->value('reporting_year');

        if ($minYear && $defaultYear) {
            // Convert the years to integers
            $startYear = (int) $minYear;
            $endYear = (int) $defaultYear;

            // Fetch the records within the range of years
            $reporting_years = ReportingYear::whereBetween('reporting_year', [$startYear, $endYear])
                ->where('status', 'active')
                ->orderBy('reporting_year', 'asc')
                ->get(['id', 'reporting_year']);

            return $reporting_years;
        }
        return [];
    }

    public function create()
    {
        try {
            $values_types = IndicatorValueType::getAllValues();
        } catch (QueryException $e) {
            $logData['file'] = __FILE__;
            $logData['method'] = __FUNCTION__;
            $logData['query'] = $e->getSql();
            $logData['exception'] = $e->getMessage();
            // Log exception to file with file name and method information
            CommonHelper::dataActionLogger($logData, 'exception');
            $values_types = [];
        }

        $goals = CommonHelper::getGoalData('all');
        if (!$goals) {
            $goals = [];
        }

        try {
            $reporting_years = $this->getYearsRange();
        } catch (QueryException $e) {
            $logData['file'] = __FILE__;
            $logData['method'] = __FUNCTION__;
            $logData['query'] = $e->getSql();
            $logData['exception'] = $e->getMessage();
            // Log exception to file with file name and method information
            CommonHelper::dataActionLogger($logData, 'exception');
            $reporting_years = [];
        }

        $departments = CommonHelper::getDepartments('all');
        if (!$departments) {
            $departments = [];
        }

        $tags = Tag::Select('tag')->distinct()->get();
        $indicator = [];

        return view('admin_pages.indicator.create', [
            'values_types' => $values_types,
            'goals' => $goals,
            'reporting_years' => $reporting_years,
            'departments' => $departments,
            'indicator' => $indicator,
            'tags' => $tags,
            'edit' => false
        ]);
    }

    public function store(IndicatorRequest $request)
    {
        $dif = 'Yes';
        $utif = 'No';
        $department = null;
        if ($request->indicator_type == 'utif') {
            $dif = 'No';
            $utif = 'Yes';
        }
        if ($request->department == null) {
            $department = null;
        } else {
            $department = $request->department;
        }
        if (!empty($request->baseline_year)) {
            $baseline = ReportingYear::where('id', $request->baseline_year)->first();
            $year = $baseline->reporting_year;
        }
        try {
            DB::beginTransaction();
            $indicator = Indicator::create([
                'indicator_number' => trim($request->indicator_number),
                'indicator_name' => trim($request->indicator_name),
                'unit' => trim($request->unit),
                'department_id' => $department,
                'goal_id' => $request->goal,
                'target_id' => $request->target,
                'is_dif' => $dif,
                'is_utif' => $utif,
                'is_positive_indicator' => trim($request->marker),
                'baseline_year_id' => $request->baseline_year,
                'baseline_year' => $year,
                'date_added' => now(),
                'date_updated' => now(),
            ]);
            if (isset($indicator) && !empty($indicator)) {
                $indicator_id = $indicator->id;
                $tags = $request->tags;
                if (!empty($tags) && count($tags) > 0) {
                    $tags = array_unique($tags);
                    if (!empty($tags)) {
                        foreach ($tags as $tag) {
                            Tag::create([
                                'tag' => trim($tag),
                                'entity_type' => 'indicator',
                                'goal_id' => $request->goal,
                                'target_id' => $request->target,
                                'indicator_id' => $indicator_id,
                                'date_added' => now()
                            ]);
                        }
                    }
                }

                $query = Indicator::query();
                $query->join('goals', 'indicators.goal_id', '=', 'goals.id');
                $query->join('targets', 'indicators.target_id', '=', 'targets.target_id');
                $query->select('indicators.id as indicator_id', 'indicators.indicator_number as indicator_number', 'indicators.indicator_name as indicator_name', 'goals.goal_name as goal_name', 'targets.target_name as target_name', 'indicators.is_dif as dif', 'indicators.is_utif as utif', 'indicators.unit as unit', 'indicators.is_positive_indicator as indicator_marker', 'indicators.status as status', 'indicators.baseline_year as baseline_year', 'indicators.date_added as date_added', 'indicators.date_updated as date_updated');
                $query->where('indicators.id', $indicator_id);
                $query->orderBy('indicators.id', 'asc');
                $indicator_data = $query->first();

                $baselineYear = ReportingYear::where('id', $request->baseline_year)->first();
                $baselineYear = $baselineYear->reporting_year;
                // Split the baseline year into start and end years
                list(
                    $startYear, $endYear
                ) = explode('-', $baselineYear);

                $query = ReportingYear::query();
                $query->where(function ($query) use ($startYear, $endYear) {
                    $query->whereRaw("CAST(SUBSTRING_INDEX(reporting_year, '-', 1) AS UNSIGNED) >= ?", [$startYear])
                        ->orWhereRaw("CAST(SUBSTRING_INDEX(reporting_year, '-', -1) AS UNSIGNED) >= ?", [$endYear]);
                });
                $query->where('status', 'active');
                $query->select('id', 'reporting_year');
                $reportingYears = $query->get();

                $value_type = null;
                if ($request->indicator_type == 'dif') {
                    $query = District::query();
                    $query->where('status', 'Active');
                    $allDistricts = $query->orderBy('sort_order', 'asc')->get();

                    if (strtolower($request->unit) == 'absolute') {
                        $value_type = $request->value_type;
                    }
                    foreach ($allDistricts as $district) {
                        foreach ($reportingYears as $reportingYear) {

                            $status = 'Pending';
                            $base_line_value = null;

                            if ($reportingYear->id == (int)$request->baseline_year && !empty($request->indicator_value)) {
                                $base_line_value = $request->indicator_value;
                                $status = 'Approved';
                            }

                            IndicatorDataEntry::create([
                                'indicator_id' => $indicator_id,
                                'indicator_number' => $indicator_data->indicator_number,
                                'indicator_name' => $indicator_data->indicator_name,
                                'target_id' => $request->target,
                                'goal_id' => $request->goal,
                                'target_name' => $indicator_data->target_name,
                                'goal_name' => $indicator_data->goal_name,
                                'reporting_year_id' => $reportingYear,
                                'unit' => $request->unit,
                                'value_type' => $value_type,
                                'target_value' => $request->target_value,
                                'indicator_value' => $base_line_value,
                                'baseline_year_id' => $request->baseline_year,
                                'baseline_year' => $baselineYear,
                                'reporting_year_id' => $reportingYear->id,
                                'reporting_year' => $reportingYear->reporting_year,
                                'district_id' => $district->id,
                                'district_name' => $district->district_name,
                                'status' => $status,
                                'created_at' => now(),
                            ]);
                        }
                    }
                } else {
                    $query = Department::query();
                    $department_data = $query->where('id', $department)
                        ->select('id', 'department_name')
                        ->first();
                    if (isset($department_data) && !empty($department_data)) {
                        foreach ($reportingYears as $reportingYear) {

                            $status = 'Pending';
                            $base_line_value = null;

                            if ($reportingYear->id == $request->baseline_year && !empty($request->indicator_value)) {
                                $base_line_value = $request->indicator_value;
                                $status = 'Approved';
                            }

                            if (strtolower($request->unit) == 'absolute') {
                                $value_type = $request->value_type;
                            }

                            IndicatorDataEntry::create([
                                'indicator_id' => $indicator_id,
                                'indicator_number' => $indicator_data->indicator_number,
                                'indicator_name' => $indicator_data->indicator_name,
                                'target_id' => $request->target,
                                'target_name' => $indicator_data->target_name,
                                'goal_id' => $request->goal,
                                'goal_name' => $indicator_data->goal_name,
                                'unit' => $request->unit,
                                'value_type' => $request->value_type,
                                'target_value' => $request->target_value,
                                'indicator_value' => $base_line_value,
                                'baseline_year_id' => $request->baseline_year,
                                'baseline_year' => $baselineYear,
                                'reporting_year_id' => $reportingYear->id,
                                'reporting_year' => $reportingYear->reporting_year,
                                'department_id' => $department_data->id,
                                'department_name' => $department_data->department_name,
                                'status' => $status,
                                'created_at' => now(),
                            ]);
                        }
                    }
                }
                DB::commit();
                return redirect('indicators')->with('success', 'Indicator Created Successfully');
            } else {
                return redirect('indicators')->with('error', 'System encounter error while saving indicator, Please try after sometime.');
            }
        } catch (QueryException $e) {
            $logData['file'] = __FILE__;
            $logData['method'] = __FUNCTION__;
            $logData['query'] = $e->getSql();
            $logData['exception'] = $e->getMessage();
            // Log exception to file with file name and method information
            CommonHelper::dataActionLogger($logData, 'exception');
            DB::rollback();
            return redirect('indicators')->with('error', 'System encounter error while saving indicator, Please try after sometime.');
        } catch (Error $e) {
            $file = __FILE__;
            $method = __FUNCTION__;
            // Log exception to file with file name and method information
            Log::channel('exception_log')->critical('Error', [
                'exception' => $e->getMessage(),
                'file' => $file,
                'method' => $method,
            ]);
            DB::rollback();
            return redirect('indicators')->with('error', 'System encounter error while saving indicator, Please try after sometime.');
        }
    }

    /*START:
    * Function: exportIndicators
    * Purpose: Export the indicators
    * Description:
    * 1. Export the indicators based on the request
    * ------------------------------------------------------------------------
    * Parameters:
    * @param Request $request
    * ------------------------------------------------------------------------
    * Returns:
    * @return Response
    * ------------------------------------------------------------------------
    * AUTHOR: SUHEL KHAN
    * DATE: 22-MAY-2024
    * ------------------------------------------------------------------------
    */

    public function edit(String $indicator_id)
    {
        $values_types = [];
        $goals = [];
        $reporting_years = [];
        $departments = [];
        $tags = [];
        $indicators = [];

        try{
            $indicators = Indicator::where('id', $indicator_id)->first();
        }catch (QueryException $e) {
            $logData['file'] = __FILE__;
            $logData['method'] = __FUNCTION__;
            $logData['query'] = $e->getSql();
            $logData['exception'] = $e->getMessage();
            // Log exception to file with file name and method information
            CommonHelper::dataActionLogger($logData, 'exception');
            $indicators = [];
        }

        if (empty($indicators)) {
            return redirect()->back()->with('error', 'Indicator - ' . $indicator_id . ' is not applicable to edit.');
        }

        try {
            $indicator_data_entries = IndicatorDataEntry::Select('indicator_id')->where('indicator_id', $indicators->id)->where('baseline_year_id', '!=', $indicators->baseline_year_id)->whereNotNull('indicator_value')->get();
        } catch (QueryException $e) {
            $logData['file'] = __FILE__;
            $logData['method'] = __FUNCTION__;
            $logData['query'] = $e->getSql();
            $logData['exception'] = $e->getMessage();
            // Log exception to file with file name and method information
            CommonHelper::dataActionLogger($logData, 'exception');
        }

        if (!empty($indicator_data_entries) && $indicator_data_entries->count() > 0) {
            return redirect()->back()->with('error', 'Indicator - ' . $indicator_id . ' is not applicable to edit because data entry exist for this indicator.');
        } else {
            try {
                $values_types = IndicatorValueType::getAllValues();
            } catch (QueryException $e) {
                $logData['file'] = __FILE__;
                $logData['method'] = __FUNCTION__;
                $logData['query'] = $e->getSql();
                $logData['exception'] = $e->getMessage();
                // Log exception to file with file name and method information
                CommonHelper::dataActionLogger($logData, 'exception');
                $values_types = [];
            }

            $goals = CommonHelper::getGoalData('all');
            if (!$goals) {
                $goals = [];
            }

            $query = ReportingYear::query();
            $query->select('reporting_year', 'id');
            $query->orderBy('reporting_year', 'asc');
            try {
                $reporting_years = $query->get();
            } catch (QueryException $e) {
                $logData['file'] = __FILE__;
                $logData['method'] = __FUNCTION__;
                $logData['query'] = $e->getSql();
                $logData['exception'] = $e->getMessage();
                // Log exception to file with file name and method information
                CommonHelper::dataActionLogger($logData, 'exception');
                $reporting_years = [];
            }

            $departments = CommonHelper::getDepartments('all');
            if (!$departments) {
                $departments = [];
            }

            try{
                $indicatorDataEntries = IndicatorDataEntry::Select('indicator_value', 'target_value', 'value_type')->where('indicator_id', $indicators->id)->where('baseline_year_id', $indicators->baseline_year_id)->whereNotNull('indicator_value')->first();
                if (isset($indicatorDataEntries) && !empty($indicatorDataEntries)) {
                    $indicators->indicator_value = $indicatorDataEntries->indicator_value;
                    $indicators->target_value = $indicatorDataEntries->target_value;
                    $indicators->value_type = $indicatorDataEntries->value_type;
                }
            } catch (QueryException $e) {
                $logData['file'] = __FILE__;
                $logData['method'] = __FUNCTION__;
                $logData['query'] = $e->getSql();
                $logData['exception'] = $e->getMessage();
                // Log exception to file with file name and method information
                CommonHelper::dataActionLogger($logData, 'exception');
                $indicatorDataEntries = [];
            }
            if (!empty($indicators) && $indicators->count() > 0) {

                $targets = CommonHelper::getTargets('all', $indicators->goal_id);
                if (!$targets) {
                    $targets = [];
                }
                $tags = Tag::where('indicator_id', $indicator_id)->get();

                return view('admin_pages.indicator.create', [
                    'values_types' => $values_types,
                    'goals' => $goals,
                    'reporting_years' => $reporting_years,
                    'departments' => $departments,
                    'indicator' => $indicators,
                    'tags' => $tags,
                    'targets' => $targets,
                    'edit' => true
                ]);
            }
        }
    }
    /*END:
    * Function: exportIndicators
    * ------------------------------------------------------------------------
    */

    /*START:
    * Function: update
    * Purpose: Update the indicator
    * Description:
    * 1. Update the indicator based on the request
    * ------------------------------------------------------------------------
    * Parameters:
    * @param UpdateIndicatorRequest $request
    * @param Indicator $indicator
    * ------------------------------------------------------------------------
    * Returns:
    * @return Response
    * ------------------------------------------------------------------------
    * AUTHOR: SUHEL KHAN
    * DATE: 22-MAY-2024
    * ------------------------------------------------------------------------
    */
    public function update(UpdateIndicatorRequest $request, Indicator $indicator)
    {
        try {
            DB::beginTransaction();
            if (!empty($indicator->id)) {
                Indicator::where('id', $indicator->id)
                    ->update([
                        // 'indicator_number' => trim($request->indicator_number),
                        // 'indicator_name' => trim($request->indicator_name),
                        // 'unit' => trim($request->unit),
                        // 'goal_id' => $request->goal,
                        'target_id' => trim($request->target),
                        // 'is_positive_indicator' => trim($request->marker),
                        // 'min_value' => $request->min_value,
                        // 'max_value' => $request->max_value,
                        'date_updated' => now(),
                    ]);

                // Update the fields
                $target = CommonHelper::getTargets($request->target);
                if (!$target) {
                    $target = [];
                }
                IndicatorDataEntry::where('indicator_id', $request->indicator_id)
                    ->update([
                        // 'indicator_number' => $request->indicator_number,
                        // 'indicator_name' => $request->indicator_name,
                        // 'goal_id' => $request->goal,
                        // 'goal_name' => $request->goal_name,
                        // 'unit' => trim($request->unit),
                        'target_id' => trim($request->target),
                        'target_name' => trim($target['target_name']),
                        'target_value' => trim($request->target_value),
                        'value_type' => trim($request->value_type),
                    ]);

                // After successful update, update the indicator_value
                IndicatorDataEntry::where('indicator_id', $request->indicator_id)
                    ->where('reporting_year_id', $indicator->baseline_year_id)
                    ->update(['indicator_value' => trim($request->indicator_value)]);
                DB::commit();
                return redirect('indicators')->with('success', 'Indicator Updated Successfully');
            }
        } catch (QueryException $e) {
            $logData['file'] = __FILE__;
            $logData['method'] = __FUNCTION__;
            $logData['query'] = $e->getSql();
            $logData['exception'] = $e->getMessage();
            // Log exception to file with file name and method information
            CommonHelper::dataActionLogger($logData, 'exception');
            DB::rollback();
            return redirect('indicators')->with('error', 'System encounter error while updating indicator, Please try after sometime.');
        }
        return redirect('indicators')->with('error', 'System encounter error while updating indicator, Please try after sometime.');
    }
    /*END:
    * Function: update
    * ------------------------------------------------------------------------
    */

    /*START:
    * Function: destroy
    * Purpose: Deactivate or Activate the indicator
    * Description:
    * 1. Deactivate or Activate the indicator based on the action
    * ------------------------------------------------------------------------
    * Parameters:
    * @param string $indicatorId
    * @param string $discontinue
    * ------------------------------------------------------------------------
    * Returns:
    * @return Response
    * ------------------------------------------------------------------------
    * AUTHOR: SUHEL KHAN
    * DATE: 22-MAY-2024
    * ------------------------------------------------------------------------
    */
    public function destroy(string $indicatorId, string $discontinue)
    {
        $query = Indicator::query();
        try {
            $indicator = $query->find($indicatorId);
            if (isset($indicator) && !empty($indicator)) {
                if (strtolower(trim($discontinue)) == 'yes') {
                    $indicator->is_discontinue = 'Yes';
                    $indicator->save();
                    return redirect('indicators')->with('success', 'Indicator discontinued successfully.');
                } else if (strtolower(trim($discontinue)) == 'no') {
                    $indicator->is_discontinue = 'No';
                    $indicator->save();
                    return redirect('indicators')->with('success', 'Indicator activated successfully.');
                } else {
                    return redirect('indicators')->with('error', 'Invalid action.');
                }
            } else {
                return redirect('indicators')->with('error', 'Indicator not found.');
            }
        } catch (QueryException $e) {
            $logData['file'] = __FILE__;
            $logData['method'] = __FUNCTION__;
            $logData['query'] = $e->getSql();
            $logData['exception'] = $e->getMessage();
            // Log exception to file with file name and method information
            CommonHelper::dataActionLogger($logData, 'exception');
            return redirect('indicators')->with('error', 'Error while Updating status of Indicator. Please try after sometime.');
        }
    }
    /*END:
    * Function: destroy
    * ------------------------------------------------------------------------
    */

    public function show($id)
    {
        // Logic for showing a specific target
    }

    /*START:
    * Function: exportIndicators
    * Purpose: Export the list of indicators in excel format
    * Description:
    * 1. Fetch the list of indicators from the database
    * 2. Export the list of indicators in excel format
    * 3. Download the excel file
    * ------------------------------------------------------------------------
    * Parameters:
    * @param Request $request
    * ------------------------------------------------------------------------
    * Returns:
    * @return Response
    * ------------------------------------------------------------------------
    * AUTHOR: SUHEL KHAN
    * DATE: 22-MAY-2024
    * ------------------------------------------------------------------------
    */
    public function exportIndicators(Request $request)
    {

        $query = Indicator::query();

        $query->select(
            'indicators.id as indicator_id',
            'indicators.indicator_number as indicator_number',
            'indicators.indicator_name as indicator_name',
            'goals.goal_name as goal_name',
            DB::raw("CONCAT(targets.target_number, ' ', targets.target_name) as target_name"),
            'indicators.is_dif as dif',
            'indicators.is_utif as utif',
            'indicators.unit as unit',
            'indicators.is_positive_indicator as indicator_marker',
            'indicators.status as status',
            'indicators.baseline_year as baseline_year',
            'indicators.date_added as date_added',
            'indicators.date_updated as date_updated'
        )
            ->join('goals', 'indicators.goal_id', '=', 'goals.id')
            ->join('targets', 'indicators.target_id', '=', 'targets.target_id')
            ->orderBy('indicators.id', 'asc');

        if ($request->has('goal') && !empty($request->input('goal'))) {
            $query->where('indicators.goal_id', $request->input('goal'));
        }

        if ($request->has('target') && !empty($request->input('target'))) {
            $query->where('indicators.target_id', $request->input('target'));
        }

        if ($request->has('type') && !empty($request->input('type'))) {
            if ($request->input('type') == 'dif') {
                $query->where('indicators.is_dif', 'Yes');
            } else if ($request->input('type') == 'utif') {
                $query->where('indicators.is_utif', 'Yes');
            }
        }

        try {
            $results = $query->get();
        } catch (QueryException $e) {
            $logData['file'] = __FILE__;
            $logData['method'] = __FUNCTION__;
            $logData['query'] = $e->getSql();
            $logData['exception'] = $e->getMessage();
            // Log exception to file with file name and method information
            CommonHelper::dataActionLogger($logData, 'exception');
            return redirect('indicators')->with('error', 'Error while exporting Targets. Please try after sometime.');
        }

        if ($results->isEmpty()) {
            return redirect('indicators')->with('error', 'No Indicator found to export.');
        }

        $filename = 'list_of_indicators_' . now()->format('Y-m-d_H-i-s') . '.xlsx';
        Excel::store(new IndicatorExport($results), 'export/indicators/' . $filename);

        // Generate the full path to the saved file
        $filePath = storage_path('app/export/indicators/' . $filename);

        return response()->download($filePath, $filename);
    }
    /*END:
    * Function: exportIndicators
    * ------------------------------------------------------------------------
    */

    /*START:
    * Function: getTargets
    * Purpose: Get the list of targets based on the goal id
    * Description:
    * 1. Fetch the list of targets based on the goal id
    * 2. Return the list of targets in json format
    * ------------------------------------------------------------------------
    * Parameters:
    * @param Request $request
    * ------------------------------------------------------------------------
    * Returns:
    * @return Response
    * ------------------------------------------------------------------------
    * AUTHOR: SUHEL KHAN
    * DATE: 22-MAY-2024
    * ------------------------------------------------------------------------
    */
    public function getTargets(Request $request)
    {
        $targets = Target::where('goal_id', $request->goal_id)->get();
        return response()->json($targets);
    }
    /*END:
    * Function: getTargets
    * ------------------------------------------------------------------------
    */
}
