<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Helpers\CommonHelper;
use App\Models\IndicatorDataEntry;
use Illuminate\Database\QueryException;
use Illuminate\Support\Facades\Log;
use Maatwebsite\Excel\Facades\Excel;
use App\Exports\DistrictWiseSdgIndicatorExport;
use App\Exports\DepartmentWiseSdgIndicatorExport;
use App\Exports\FinancialYearWiseSdgIndicatorExport;
use App\Models\ReportingYear;
use PDF;

class ExportReportController extends Controller
{
    function __construct()
    {
        $this->middleware('permission:View Export Data', ['only' => ['index', 'exportData']]);
        $this->middleware('auth');
    }

    public function index()
    {
        $logData = array();
        $financial_year_wise_data = array();
        $district_wise_data = array();
        $department_wise_data = array();
        $filter_financial_year = array();
        $filter_goal = array();
        $filter_district = array();
        $filter_department = array();
        $indicator_type = '';
        //prepared the log file name and location
        $file_name = date('Y-m-d') . '_export_report.txt';
        $location = 'logs/info-log/' . auth()->user()->email . '/export-report';
        $logData['user'] = auth()->user()->email;
        $logData['file'] = __FILE__;
        $logData['method'] = __FUNCTION__;
        $logData['information'] = "User accessed the Export Report page.";
        //using dataActionLogger function to log the information
        CommonHelper::dataActionLogger($logData, 'info', $location, $file_name);

        $user = auth()->user();
        $districts = [];
        $departments = [];

        $reportingYears = CommonHelper::getReoprtingYears();

        if (isset($reportingYears) && !empty($reportingYears)) {
            $financialYear = $reportingYears['district_indicator_pair'];
        } else {
            $financialYear = [];
        }

        $goals = CommonHelper::getGoalData('all');
        if (!$goals) {
            $goals = [];
        }

        // Check the user role and fetch the districts and departments based on the user role
        if ($user->hasRole('Super Admin') || $user->hasRole('Admin')) {
            // if the user is Super Admin or Admin, then fetch all the districts and departments
            $districts = CommonHelper::getDistricts('all');
            if (!$districts) {
                $districts = [];
            }

            $departments = CommonHelper::getDepartments('all');
            if (!$departments) {
                $departments = [];
            }
        } elseif ($user->hasRole('District-User') || $user->hasRole('District-Approver') || $user->hasRole('District-HOD')) {
            // if the user is District-User or District-Approver, then fetch the districts based on the user's district
            $districts = CommonHelper::getDistricts($user->district_id);
            if (!$districts) {
                $districts = [];
            }
        } elseif ($user->hasRole('Department-User') || $user->hasRole('Department-Approver') || $user->hasRole('Department-HOD')) {
            // if the user is Department-User or Department-Approver, then fetch the departments based on the user's department
            $departments = CommonHelper::getDepartments($user->department_id);
            if (!$departments) {
                $departments = [];
            }
        }

        $default_year = CommonHelper::getDefaultYear();
        if (!$default_year) {
            $default_year = [];
        }

        return view('admin_pages.export-report.index', [
            'districts' => $districts, 'departments' => $departments, 'financialYears' => $financialYear, 'goals' => $goals, 'user' => $user,
            'financial_year_wise_data' => $financial_year_wise_data, 'district_wise_data' => $district_wise_data,
            'department_wise_data' => $department_wise_data, 'indicator_type' => $indicator_type, 'error' => '', 'default_year' => $default_year,
            'filter_financial_year' => $filter_financial_year, 'filter_goal' => $filter_goal, 'filter_district' => $filter_district, 'filter_department' => $filter_department
        ]);
    }

    public function exportData(Request $request)
    {
        ini_set('memory_limit', '1024M');
        $financial_year_wise_data = array();
        $district_wise_data = array();
        $department_wise_data = array();
        $financial_year_wise_indicator_count = array();
        $financial_year_wise_target_count = array();
        $district_wise_indicator_count = array();
        $district_wise_target_count = array();
        $department_wise_indicator_count = array();
        $department_wise_target_count = array();
        $indicator_type = '';
        $error = '';
        //prepared the log file name and location
        $file_name = date('Y-m-d') . '_export_report.txt';
        $location = 'logs/info-log/' . auth()->user()->email . '/export-report';
        $logData['user'] = auth()->user()->email;
        $logData['file'] = __FILE__;
        $logData['method'] = __FUNCTION__;
        $logData['information'] = "User requested to export the data.";
        //using dataActionLogger function to log the information
        CommonHelper::dataActionLogger($logData, 'info', $location, $file_name);

        $user = auth()->user();
        $districts = [];
        $departments = [];

        $reportingYears = CommonHelper::getReoprtingYears();

        if (isset($reportingYears) && !empty($reportingYears)) {
            $financialYear1 = $reportingYears['district_indicator_pair'];
        } else {
            $financialYear1 = [];
        }

        $goals = CommonHelper::getGoalData('all');
        if (!$goals) {
            $goals = [];
        }

        // Check the user role and fetch the districts and departments based on the user role
        if ($user->hasRole('Super Admin') || $user->hasRole('Admin')) {
            // if the user is Super Admin or Admin, then fetch all the districts and departments
            $districts = CommonHelper::getDistricts('all');
            if (!$districts) {
                $districts = [];
            }

            $departments = CommonHelper::getDepartments('all');
            if (!$departments) {
                $departments = [];
            }
        } elseif ($user->hasRole('District-User') || $user->hasRole('District-Approver') || $user->hasRole('District-HOD')) {
            // if the user is District-User or District-Approver, then fetch the districts based on the user's district
            $districts = CommonHelper::getDistricts($user->district_id);
            if (!$districts) {
                $districts = [];
            }
        } elseif ($user->hasRole('Department-User') || $user->hasRole('Department-Approver') || $user->hasRole('Department-HOD')) {
            // if the user is Department-User or Department-Approver, then fetch the departments based on the user's department
            $departments = CommonHelper::getDepartments($user->department_id);
            if (!$departments) {
                $departments = [];
            }
        }

        $base = $request->base;
        $flexRadioDefault = $request->flexRadioDefault1;
        $district = '';
        $department = '';
        $financialYear = '';
        $goal = '';
        if ($request->base == 'financial-year-wise') {
            // check validation on radio button in case of financial year tab
            if (!isset($flexRadioDefault) && empty($flexRadioDefault)) {
                $error = 'Select by District or Department is required.';
            }

            if (empty($error)) {
                // to check if radio type is district then fetch data by district else return response.
                if ($flexRadioDefault == 'district') {
                    if ($user->hasRole('District-User') || $user->hasRole('District-Approver') || $user->hasRole('District-HOD')) {
                        $district = $user->district_id;
                    } else {
                        $district = $request->district1;
                        if (!isset($district) && empty($district)) {
                            $error = 'District is required';
                        }
                    }
                    $department = '';
                } else {
                    $district = '';
                    if ($user->hasRole('Department-User') || $user->hasRole('Department-Approver') || $user->hasRole('Department-HOD')) {
                        $department = $user->department_id;
                    } else {
                        $department = $request->department1;
                        if (!isset($department) && empty($department)) {
                            $error = 'Department is required';
                        }
                    }
                }
            }

            $financialYear = $request->financial_year1;
            if (!isset($financialYear) && empty($financialYear) && empty($error)) {
                $error = 'Financial Year is required';
            }

            $goal = $request->goal1;
            if (!isset($goal) && empty($goal) && empty($error)) {
                $error = 'Goal is required';
            }
        } elseif ($base == 'district-wise') {
            // to check if district is not empty.
            $district = $request->district3;
            if (!isset($district) && empty($district) && empty($error)) {
                $error = 'District is required';
            }

            $financialYear = $request->financial_year3;
            if (!isset($financialYear) && empty($financialYear) && empty($error)) {
                $error = 'Financial Year is required';
            }

            $goal = $request->goal3;
            if (!isset($goal) && empty($goal) && empty($error)) {
                $error = 'Goal is required';
            }
        } elseif ($base == 'department-wise') {
            // to check if department is not empty.
            $department = $request->department2;
            if (!isset($department) && empty($department) && empty($error)) {
                $error = 'Department is required';
            }

            $financialYear = $request->financial_year2;
            if (!isset($financialYear) && empty($financialYear) && empty($error)) {
                $error = 'Financial Year is required';
            }

            $goal = $request->goal2;
            if (!isset($goal) && empty($goal) && empty($error)) {
                $error = 'Goal is required';
            }
        } else {
            $error = 'System cannot identify the tab of the report, it must be "Financial Year Wise" or "District Wise" or "Department Wise".';
        }
        $filter_financial_year = $financialYear;
        $filter_goal = $goal;
        $filter_district = $district;
        $filter_department = $department;

        if (empty($error)) {

            /*SELECT
                'ide.goal_id AS goal_id',
                'indicator_data_entries.goal_name AS goal_name',
                'indicator_data_entries.target_name AS target_name',
                'indicator_data_entries.indicator_number AS indicator_number',
                'indicator_data_entries.indicator_name AS indicator_name',
                'indicator_data_entries.unit AS unit',
                'ide_baseline.reporting_year AS baseline_year',
                'ide_baseline.indicator_normalised_value AS baseline_value',
                'indicator_data_entries.reporting_year AS reporting_year',
                'indicator_data_entries.target_value AS target_value',
                'indicator_data_entries.district_name AS district_name',
                'indicator_data_entries.indicator_normalised_value AS achieved_value'
            FROM
                indicator_data_entries ide
                LEFT JOIN indicator_data_entries ide_baseline ON ide.goal_id = ide_baseline.goal_id
                    AND ide.target_id = ide_baseline.target_id
                    AND ide.indicator_id = ide_baseline.indicator_id
                    AND ide.district_id = ide_baseline.district_id
                    AND ide_baseline.reporting_year = ide.baseline_year
            WHERE
                ide.district_id IS NOT NULL
                AND ide.reporting_year IN ('2020-2021', '2021-2022', '2022-2023')
                AND ide.district_id IN (19) 
                AND ide.goal_id IN (1, 2) 
            ORDER BY
                ide.id
            LIMIT 100 OFFSET 0; 
            */

            //Query to count the number of indicator available in a goal
            $query = IndicatorDataEntry::query();
            // Purpose: Added the below condition as the report should be generated only for the approved data
            // Author: Almaaz Ahmed
            // Date: 10-January-2024
            $query->where('indicator_data_entries.status','Approved');
            /*SELECT
                goal_id,
                COUNT(indicator_id) AS indicator_count
            FROM
                indicator_data_entries
            WHERE
                AND reporting_year IN ('2020-2021', '2021-2022', '2022-2023')
                AND district_id IN (19) 
                AND goal_id IN (1, 2) 
            GROUP BY
                goal_id;
            */
            $indicatorCountQuery = IndicatorDataEntry::query();
            // Purpose: Added the below condition as the report should be generated only for the approved data
            // Author: Almaaz Ahmed
            // Date: 10-January-2024
            $indicatorCountQuery->where('indicator_data_entries.status','Approved');
            //Query to count the number of target available in a goal
            /*SELECT
                target_id,
                COUNT(indicator_id) AS indicator_count
            FROM
                indicator_data_entries
            WHERE
                AND reporting_year IN ('2020-2021', '2021-2022')
                AND district_id IN (19)
                AND goal_id IN (1, 2)
            GROUP BY
                target_id;
            */
            $targetCountQuery = IndicatorDataEntry::query();
            // Purpose: Added the below condition as the report should be generated only for the approved data
            // Author: Almaaz Ahmed
            // Date: 10-January-2024
            $targetCountQuery->where('indicator_data_entries.status','Approved');
            // To define the columns to be selected based on the tab of the report
            if (!empty($district)) {
                $indicator_type = 'district';
                $query->select([
                    'indicator_data_entries.id AS indicator_id',
                    'indicator_data_entries.goal_id AS goal_id',
                    'indicator_data_entries.goal_name AS goal_name',
                    'indicator_data_entries.target_id AS target_id',
                    'indicator_data_entries.target_name AS target_name',
                    'indicator_data_entries.indicator_number AS indicator_number',
                    'indicator_data_entries.indicator_name AS indicator_name',
                    'indicator_data_entries.unit AS unit',
                    'ide_baseline.reporting_year AS baseline_year',
                    'ide_baseline.indicator_normalised_value AS baseline_value',
                    'indicator_data_entries.reporting_year AS reporting_year',
                    'indicator_data_entries.target_value AS target_value',
                    'indicator_data_entries.district_name AS district_name',
                    'indicator_data_entries.district_id AS district_id',
                    'indicator_data_entries.indicator_normalised_value AS achieved_value'
                ]);
            } else if (!empty($department)) {
                $indicator_type = 'department';
                $query->select([
                    'indicator_data_entries.id AS indicator_id',
                    'indicator_data_entries.goal_id AS goal_id',
                    'indicator_data_entries.goal_name AS goal_name',
                    'indicator_data_entries.target_id AS target_id',
                    'indicator_data_entries.target_name AS target_name',
                    'indicator_data_entries.indicator_number AS indicator_number',
                    'indicator_data_entries.indicator_name AS indicator_name',
                    'indicator_data_entries.unit AS unit',
                    'ide_baseline.reporting_year AS baseline_year',
                    'ide_baseline.indicator_value AS baseline_value',
                    'indicator_data_entries.reporting_year AS reporting_year',
                    'indicator_data_entries.target_value AS target_value',
                    'indicator_data_entries.department_name AS department_name',
                    'indicator_data_entries.indicator_value AS achieved_value'
                ]);
            } else {
                $error = 'System cannot identify the tab of the report, it must be "Financial Year Wise" or "District Wise" or "Department Wise".';
            }
            // to count the number of indicators for each goal
            $indicatorCountQuery->selectRaw('goal_id, COUNT(indicator_id) as indicator_count')
                ->groupBy('goal_id');

            // to count the number of targets for each goal
            $targetCountQuery->selectRaw('target_id, COUNT(indicator_id) as indicator_count')
                ->groupBy('target_id');

            if ($indicator_type == 'department') {
                // to join the indicator_data_entries table with itself to get the baseline value
                $query->leftJoin('indicator_data_entries AS ide_baseline', function ($join) {
                    $join->on('indicator_data_entries.goal_id', '=', 'ide_baseline.goal_id')
                        ->on('indicator_data_entries.target_id', '=', 'ide_baseline.target_id')
                        ->on('indicator_data_entries.indicator_id', '=', 'ide_baseline.indicator_id')
                        ->on('indicator_data_entries.department_id', '=', 'ide_baseline.department_id')
                        ->on('ide_baseline.reporting_year', '=', 'indicator_data_entries.baseline_year');
                });
            } else {
                // to join the indicator_data_entries table with itself to get the baseline value
                $query->leftJoin('indicator_data_entries AS ide_baseline', function ($join) {
                    $join->on('indicator_data_entries.goal_id', '=', 'ide_baseline.goal_id')
                        ->on('indicator_data_entries.target_id', '=', 'ide_baseline.target_id')
                        ->on('indicator_data_entries.indicator_id', '=', 'ide_baseline.indicator_id')
                        ->on('indicator_data_entries.district_id', '=', 'ide_baseline.district_id')
                        ->on('ide_baseline.reporting_year', '=', 'indicator_data_entries.baseline_year');
                });
            }

            if (!empty($district)) {
                if (is_array($district) && count($district) > 0 && strtolower($district[0]) != 'all') {
                    $query->whereIn('indicator_data_entries.district_id', $district);
                    $indicatorCountQuery->whereIn('district_id', $district);
                    $targetCountQuery->whereIn('district_id', $district);
                    if (isset($request->export) && !empty($request->export) && $request->export == 'pdf') {
                        if (count($district) > 2) {
                            $response['message'] = 'Please select maximum 2 districts to generate the PDF report.';
                            $response['status'] = 'error';
                            return response()->json($response);
                        }
                    }
                } else if (!is_array($district)) {
                    $query->where('indicator_data_entries.district_id', $district);
                    $indicatorCountQuery->where('district_id', $district);
                    $targetCountQuery->where('district_id', $district);
                } else {
                    $query->whereNotNull('indicator_data_entries.district_id');
                    $indicatorCountQuery->whereNotNull('district_id');
                    $targetCountQuery->whereNotNull('district_id');
                    if (isset($request->export) && !empty($request->export) && $request->export == 'pdf') {
                        if (count($district) > 3) {
                            $response['message'] = 'Please select maximum 2 districts to generate the PDF report.';
                            $response['status'] = 'error';
                            return response()->json($response);
                        }
                    }
                }
            } else if (!empty($department)) {
                if (is_array($department) && count($department) > 0 && strtolower($department[0]) != 'all') {
                    $query->whereIn('indicator_data_entries.department_id', $department);
                    $indicatorCountQuery->whereIn('department_id', $department);
                    $targetCountQuery->whereIn('department_id', $department);
                } else if (!is_array($department)) {
                    $query->where('indicator_data_entries.department_id', $department);
                    $indicatorCountQuery->where('department_id', $department);
                    $targetCountQuery->where('department_id', $department);
                } else {
                    $query->whereNotNull('indicator_data_entries.department_id');
                    $indicatorCountQuery->whereNotNull('department_id');
                    $targetCountQuery->whereNotNull('department_id');
                }
            } else if (!empty($flexRadioDefault)) {
                if ($flexRadioDefault == 'district') {
                    $query->whereNotNull('indicator_data_entries.district_id');
                    $indicatorCountQuery->whereNotNull('district_id');
                    $targetCountQuery->whereNotNull('district_id');
                } else {
                    $query->whereNotNull('indicator_data_entries.department_id');
                    $indicatorCountQuery->whereNotNull('department_id');
                    $targetCountQuery->whereNotNull('department_id');
                }
            }

            if (!empty($goal) && count($goal) > 0 && strtolower($goal[0]) != 'all') {
                $query->whereIn('indicator_data_entries.goal_id', $goal);
                $indicatorCountQuery->whereIn('goal_id', $goal);
                $targetCountQuery->whereIn('goal_id', $goal);
            }

            if (!empty($financialYear) && count($financialYear) > 0) {
                if (isset($request->export) && !empty($request->export) && $request->export == 'pdf') {
                    if (strtolower($financialYear[0]) == 'all' && count($financialYear) > 3) {
                        $response['message'] = 'Unsupported Financial Years! PDF can be generated only for two years at a time';
                    } else if (!empty($financialYear) && strtolower($financialYear[0]) != 'all' && count($financialYear) > 3) {
                        $response['message'] = 'Unsupported Financial Years! PDF can be generated only for two years at a time';
                    }
                    if (isset($response['message']) && !empty($response['message'])) {
                        $response['status'] = 'error';
                        return response()->json($response);
                    }
                }
                foreach ($financialYear as $key => $value) {
                    if ($value != 'all') {
                        $years = explode("-", $value);
                        $startYear = intval($years[0]);
                        $year[] = $startYear . "-" . $startYear + 1;
                    }
                }
                $query->whereIn('indicator_data_entries.reporting_year', $year);
                $indicatorCountQuery->whereIn('reporting_year', $year);
                $targetCountQuery->whereIn('reporting_year', $year);
            }

            $financial_year_wise_data_export = array();
            $district_wise_data_export = array();
            $department_wise_data_export = array();
            try {
                if ($request->base == 'financial-year-wise') {
                    if (isset($request->export) && !empty($request->export) && $request->export == 'excel' || $request->export == 'pdf') {

                        $financial_year_wise_data_export = $query->get();

                        if (count($financial_year_wise_data_export) > 0) {
                            if ($financial_year_wise_data_export->isEmpty()) {
                                $response['status'] = 'error';
                                $response['message'] = 'No data found for the selected criteria.';
                                return response()->json($response);
                            } else {
                                // to generate the pdf for financial year wise indicators
                                if ($request->export == 'pdf') {

                                    if(count($financial_year_wise_data_export) > 700){
                                        $response['status'] = 'error';
                                        $response['message'] = 'Data is too large to generate the PDF. Please remove some of the goals and generate PDF again.';
                                        return response()->json($response);
                                    }
                                    $organizedData = [];
                                    $reporting_years = [];
                                    $districts_array = [];
                                    if ($indicator_type == 'district') {
                                        foreach ($financial_year_wise_data_export as $entry) {
                                            $indicator_id = $entry->indicator_id;
                                            $districtName = $entry->district_name;
                                            $districtId = $entry->district_id;
                                            $goalName = $entry->goal_name;
                                            $indicatorName = $entry->indicator_number . ': ' . $entry->indicator_name;
                                            $baselineYear = $entry->baseline_year;
                                            $baselineValue = $entry->baseline_value;
                                            $targetValue = $entry->target_value;
                                            $reportingYear = $entry->reporting_year;
                                            $achievedValue = $entry->achieved_value;

                                            if (!isset($organizedData[$indicator_id])) {
                                                $organizedData[$indicator_id] = [
                                                    'goal' => $goalName,
                                                    'indicator' => $indicatorName,
                                                    'baseline_year' => $baselineYear,
                                                    'baseline_value' => $baselineValue,
                                                    'target_value' => $targetValue,
                                                ];
                                            }

                                            $organizedData[$indicator_id][$districtId][$reportingYear] = $achievedValue;

                                            $yearParts = explode('-', $reportingYear);
                                            $key = ($yearParts[0]) . '-' . substr($yearParts[1], 2);
                                            $reporting_years[$key] = $reportingYear;
                                            $districts_array[$districtId] = $districtName;
                                        }
                                    } else if ($indicator_type == 'department') {
                                        foreach ($financial_year_wise_data_export as $entry) {
                                            $indicator_id = $entry->indicator_id;
                                            $departmentName = $entry->department_name;
                                            $goalName = $entry->goal_name;
                                            $indicatorName = $entry->indicator_number . ': ' . $entry->indicator_name;
                                            $baselineYear = $entry->baseline_year;
                                            $baselineValue = $entry->baseline_value;
                                            $targetValue = $entry->target_value;
                                            $reportingYear = $entry->reporting_year;
                                            $achievedValue = $entry->achieved_value;

                                            if (!isset($organizedData[$indicator_id])) {
                                                $organizedData[$indicator_id] = [
                                                    'goal' => $goalName,
                                                    'indicator' => $indicatorName,
                                                    'baseline_year' => $baselineYear,
                                                    'baseline_value' => $baselineValue,
                                                    'target_value' => $targetValue,
                                                    'department' => $departmentName
                                                ];
                                            }
                                            $yearParts = explode('-', $reportingYear);
                                            $key = ($yearParts[0]) . '-' . substr($yearParts[1], 2);
                                            $reporting_years[$key] = $reportingYear;

                                            $organizedData[$indicator_id][$reportingYear] = $achievedValue;
                                        }
                                    }
                                    try {
                                        $pdf = PDF::loadView('admin_pages.export-report.indicators_pdf_report', [
                                            'indicators_data' => $organizedData,
                                            'indicator_type' => $indicator_type,
                                            'reporting_years' => $reporting_years,
                                            'districts' => $districts_array,
                                            'name' => 'Financial Year Wise Indicators Report',
                                        ]);
                                    } catch (\Exception $e) {
                                        $logData['file'] = __FILE__;
                                        $logData['method'] = __FUNCTION__;
                                        $logData['error'] = $e->getMessage();
                                        // Log error to file with file name and method information
                                        CommonHelper::dataActionLogger($logData, 'error');

                                        $response['status'] = 'error';
                                        $response['message'] = 'Error in generating the PDF. Exception: ' . $e->getMessage();
                                        return response()->json($response);
                                    } catch (\Throwable $e) {
                                        $logData['file'] = __FILE__;
                                        $logData['method'] = __FUNCTION__;
                                        $logData['exception'] = $e->getMessage();
                                        // Log error to file with file name and method information
                                        CommonHelper::dataActionLogger($logData, 'error');

                                        $response['status'] = 'error';
                                        $response['message'] = 'Error in generating the PDF. Exception: ' . $e->getMessage();
                                        return response()->json($response);
                                    }
                                    if (!$pdf && $pdf->getError()) {
                                        $response['status'] = 'error';
                                        $response['message'] = 'Error in generating the PDF.';
                                        return response()->json($response);
                                    }
                                    $filename = 'financial_year_wise_data_' . now()->format('Y-m-d_H-i-s') . '.pdf';
                                    $pdf->save(storage_path('app/export/SDG-Indicators-reports/' . $filename));
                                    $filePath = asset('storage/app/export/SDG-Indicators-reports/' . $filename);
                                    $response['status'] = 'success';
                                    $response['filepath'] = $filePath;
                                    return response()->json($response);
                                } else {
                                    $financial_year_wise_indicator_count = $indicatorCountQuery->get();
                                    $financial_year_wise_target_count = $targetCountQuery->get();
                                    // Export the data to excel file (xlsx format
                                    $filename = 'financial_year_wise_data_' . now()->format('Y-m-d_H-i-s') . '.xlsx';
                                    $success = Excel::store(new FinancialYearWiseSdgIndicatorExport($financial_year_wise_data_export, $financial_year_wise_target_count, $financial_year_wise_indicator_count, $indicator_type), 'export/SDG-Indicators-reports/' . $filename);

                                    if ($success) {
                                        // Generate the full path to the saved file
                                        $filePath = storage_path('app/export/SDG-Indicators-reports/' . $filename);
                                        $filePath = asset('storage/app/export/SDG-Indicators-reports/' . $filename);
                                        $response['status'] = 'success';
                                        $response['filepath'] = $filePath;
                                        return response()->json($response);
                                    } else {
                                        $response['status'] = 'error';
                                        $response['message'] = 'Error in exporting the data.';
                                        return response()->json($response);
                                    }
                                }
                            }
                        } else {
                            $response['status'] = 'error';
                            $response['message'] = 'No data found for the selected criteria.';
                            return response()->json($response);
                        }
                    } else {
                        $financial_year_wise_data = $query->paginate(100);
                        if ($financial_year_wise_data->total() < 1) {
                            $error = 'No data found for the selected criteria.';
                        }
                    }
                } else if ($request->base == 'district-wise') {
                    if (isset($request->export) && !empty($request->export) && $request->export == 'excel' || $request->export == 'pdf') {

                        $district_wise_data_export = $query->get();

                        if (count($district_wise_data_export) > 0) {
                            if ($district_wise_data_export->count() < 1) {
                                $response['status'] = 'error';
                                $response['message'] = 'No data found for the selected criteria.';
                                return response()->json($response);
                            } else {
                                // to generate the pdf for district wise indicators
                                if ($request->export == 'pdf') {
                                    if(count($district_wise_data_export) > 700){
                                        $response['status'] = 'error';
                                        $response['message'] = 'Data is too large to generate the PDF. Please remove some of the goals and generate PDF again.';
                                        return response()->json($response);
                                    }
                                    $organizedData = [];
                                    $reporting_years = [];
                                    $districts_array = [];
                                    foreach ($district_wise_data_export as $entry) {
                                        $indicator_id = $entry->indicator_id;
                                        $districtName = $entry->district_name;
                                        $districtId = $entry->district_id;
                                        $goalName = $entry->goal_name;
                                        $indicatorName = $entry->indicator_number . ': ' . $entry->indicator_name;
                                        $baselineYear = $entry->baseline_year;
                                        $baselineValue = $entry->baseline_value;
                                        $targetValue = $entry->target_value;
                                        $reportingYear = $entry->reporting_year;
                                        $achievedValue = $entry->achieved_value;

                                        if (!isset($organizedData[$indicator_id])) {
                                            $organizedData[$indicator_id] = [
                                                'goal' => $goalName,
                                                'indicator' => $indicatorName,
                                                'baseline_year' => $baselineYear,
                                                'baseline_value' => $baselineValue,
                                                'target_value' => $targetValue,
                                            ];
                                        }

                                        $organizedData[$indicator_id][$districtId][$reportingYear] = $achievedValue;

                                        $yearParts = explode('-', $reportingYear);
                                        $key = ($yearParts[0]) . '-' . substr($yearParts[1], 2);
                                        $reporting_years[$key] = $reportingYear;
                                        $districts_array[$districtId] = $districtName;
                                    }
                                    try {
                                        $pdf = PDF::loadView('admin_pages.export-report.indicators_pdf_report', [
                                            'indicators_data' => $organizedData,
                                            'reporting_years' => $reporting_years,
                                            'districts' => $districts_array,
                                            'indicator_type' => 'district',
                                            'name' => 'District Wise indicators PDF report'
                                        ]);
                                    } catch (\Exception $e) {
                                        $logData['file'] = __FILE__;
                                        $logData['method'] = __FUNCTION__;
                                        $logData['exception'] = $e->getMessage();
                                        // Log error to file with file name and method information
                                        CommonHelper::dataActionLogger($logData, 'error');

                                        $response['status'] = 'error';
                                        $response['message'] = 'Error in generating the PDF. Exception: ' . $e->getMessage();
                                        return response()->json($response);
                                    }
                                    if (!$pdf && $pdf->getError()) {
                                        $response['status'] = 'error';
                                        $response['message'] = 'Error in generating the PDF.';
                                        return response()->json($response);
                                    }
                                    $filename = 'district_wise_data_' . now()->format('Y-m-d_H-i-s') . '.pdf';
                                    $pdf->save(storage_path('app/export/SDG-Indicators-reports/' . $filename));
                                    $filePath = asset('storage/app/export/SDG-Indicators-reports/' . $filename);
                                    $response['status'] = 'success';
                                    $response['filepath'] = $filePath;
                                    return response()->json($response);
                                } else {
                                    $district_wise_indicator_count = $indicatorCountQuery->get();
                                    $district_wise_target_count = $targetCountQuery->get();
                                    // Export the data to excel file (xlsx format
                                    $filename = 'district_wise_data_' . now()->format('Y-m-d_H-i-s') . '.xlsx';
                                    $status = Excel::store(new DistrictWiseSdgIndicatorExport($district_wise_data_export, $district_wise_target_count, $district_wise_indicator_count), 'export/SDG-Indicators-reports/' . $filename);

                                    if ($status) {
                                        // Generate the full path to the saved file
                                        $filePath = storage_path('app/export/SDG-Indicators-reports/' . $filename);
                                        $filePath = asset('storage/app/export/SDG-Indicators-reports/' . $filename);
                                        $response['status'] = 'success';
                                        $response['filepath'] = $filePath;
                                        return response()->json($response);
                                    } else {
                                        $response['status'] = 'error';
                                        $response['message'] = 'Error in exporting the data.';
                                        return response()->json($response);
                                    }
                                }
                            }
                        } else {
                            $response['status'] = 'error';
                            $response['message'] = 'No data found for the selected criteria.';
                            return response()->json($response);
                        }
                    } else {
                        $district_wise_data = $query->paginate(100);
                        if ($district_wise_data->total() < 1) {
                            $error = 'No data found for the selected criteria.';
                        }
                    }
                } else if ($request->base == 'department-wise') {
                    if (isset($request->export) && !empty($request->export) && $request->export == 'excel' || $request->export == 'pdf') {

                        $department_wise_data_export = $query->get();

                        if (count($department_wise_data_export) > 0) {
                            if ($department_wise_data_export->count() < 1) {
                                $response['status'] = 'error';
                                $response['message'] = 'No data found for the selected criteria.';
                                return response()->json($response);
                            } else {
                                // to generate the pdf for department wise indicators
                                if ($request->export == 'pdf') {
                                    if(count($department_wise_data_export) > 700){
                                        $response['status'] = 'error';
                                        $response['message'] = 'Data is too large to generate the PDF. Please remove some of the goals and generate PDF again.';
                                        return response()->json($response);
                                    }
                                    $organizedData = [];
                                    $reporting_years = [];
                                    foreach ($department_wise_data_export as $entry) {
                                        $indicator_id = $entry->indicator_id;
                                        $departmentName = $entry->department_name;
                                        $goalName = $entry->goal_name;
                                        $indicatorName = $entry->indicator_number . ': ' . $entry->indicator_name;
                                        $baselineYear = $entry->baseline_year;
                                        $baselineValue = $entry->baseline_value;
                                        $targetValue = $entry->target_value;
                                        $reportingYear = $entry->reporting_year;
                                        $achievedValue = $entry->achieved_value;

                                        if (!isset($organizedData[$indicator_id])) {
                                            $organizedData[$indicator_id] = [
                                                'goal' => $goalName,
                                                'indicator' => $indicatorName,
                                                'baseline_year' => $baselineYear,
                                                'baseline_value' => $baselineValue,
                                                'target_value' => $targetValue,
                                                'department' => $departmentName
                                            ];
                                        }
                                        $yearParts = explode('-', $reportingYear);
                                        $key = ($yearParts[0]) . '-' . substr($yearParts[1], 2);
                                        $reporting_years[$key] = $reportingYear;

                                        $organizedData[$indicator_id][$reportingYear] = $achievedValue;
                                    }
                                    try {
                                        $pdf = PDF::loadView('admin_pages.export-report.indicators_pdf_report', [
                                            'indicators_data' => $organizedData,
                                            'reporting_years' => $reporting_years,
                                            'indicator_type' => 'department',
                                            'name' => 'Department Wise Indicators Report'
                                        ]);
                                    } catch (\Exception $e) {
                                        $logData['file'] = __FILE__;
                                        $logData['method'] = __FUNCTION__;
                                        $logData['exception'] = $e->getMessage();
                                        // Log error to file with file name and method information
                                        CommonHelper::dataActionLogger($logData, 'error');

                                        $response['status'] = 'error';
                                        $response['message'] = 'Error in generating the PDF. Exception: ' . $e->getMessage();
                                        return response()->json($response);
                                    }
                                    if (!$pdf && $pdf->getError()) {
                                        $response['status'] = 'error';
                                        $response['message'] = 'Error in generating the PDF.';
                                        return response()->json($response);
                                    }
                                    $filename = 'department_wise_data_' . now()->format('Y-m-d_H-i-s') . '.pdf';
                                    $pdf->save(storage_path('app/export/SDG-Indicators-reports/' . $filename));
                                    $filePath = asset('storage/app/export/SDG-Indicators-reports/' . $filename);
                                    $response['status'] = 'success';
                                    $response['filepath'] = $filePath;
                                    return response()->json($response);
                                } else {
                                    $department_wise_indicator_count = $indicatorCountQuery->get();
                                    $department_wise_target_count = $targetCountQuery->get();
                                    // Export the data to excel file (xlsx format
                                    $filename = 'department_wise_data_' . now()->format('Y-m-d_H-i-s') . '.xlsx';
                                    $success = Excel::store(new DepartmentWiseSdgIndicatorExport($department_wise_data_export, $department_wise_target_count, $department_wise_indicator_count), 'export/SDG-Indicators-reports/' . $filename);

                                    if ($success) {
                                        // Generate the full path to the saved file
                                        $filePath = storage_path('app/export/SDG-Indicators-reports/' . $filename);
                                        $filePath = asset('storage/app/export/SDG-Indicators-reports/' . $filename);
                                        $response['status'] = 'success';
                                        $response['filepath'] = $filePath;
                                        return response()->json($response);
                                    } else {
                                        $response['status'] = 'error';
                                        $response['message'] = 'Error in exporting the data.';
                                        return response()->json($response);
                                    }
                                }
                            }
                        } else {
                            $response['status'] = 'error';
                            $response['message'] = 'No data found for the selected criteria.';
                            return response()->json($response);
                        }
                    } else {
                        $department_wise_data = $query->paginate(100);
                        if ($department_wise_data->total() < 1) {
                            $error = 'No data found for the selected criteria.';
                        }
                    }
                }
            } catch (QueryException $e) {
                $logData['file'] = __FILE__;
                $logData['method'] = __FUNCTION__;
                $logData['query'] = $e->getSql();
                $logData['exception'] = $e->getMessage();
                // Log exception to file with file name and method information
                CommonHelper::dataActionLogger($logData, 'exception');

                $query = $query->toSql();
                $message = $e->getMessage();
                $logData['information'] = "Error in fetching the data from the database.\nException:" . $message . "\nQuery:" . $query;
                CommonHelper::dataActionLogger($logData, 'info', $location, $file_name);

                if (isset($request->export) && !empty($request->export) && $request->export == 'excel') {
                    $response['status'] = 'error';
                    $response['message'] = 'Error in fetching the data from the database.';
                    return response()->json($response);
                } else {
                    $financial_year_wise_data = array();
                    $district_wise_data = array();
                    $department_wise_data = array();
                    $financial_year_wise_indicator_count = array();
                    $financial_year_wise_target_count = array();
                    $district_wise_indicator_count = array();
                    $district_wise_target_count = array();
                    $department_wise_indicator_count = array();
                    $department_wise_target_count = array();
                    $error = 'Error in fetching the data from the database.';
                }
            }

            //added the below code to show financial on the export page
            $default_year = '';
            try{
                $default_year = ReportingYear::where('is_default','Yes')->value('reporting_year');
            }catch(QueryException $e){
                $logData['file'] = __FILE__;
                $logData['method'] = __FUNCTION__;
                $logData['query'] = $e->getSql();
                $logData['exception'] = $e->getMessage();
                // Log exception to file with file name and method information
                CommonHelper::dataActionLogger($logData, 'exception');
            }
            
            if (!$default_year) {
                $default_year = '';
            }

            return view('admin_pages.export-report.index', [
                'districts' => $districts, 'departments' => $departments, 'financialYears' => $financialYear1,
                'goals' => $goals, 'user' => $user, 'financial_year_wise_data' => $financial_year_wise_data, 'district_wise_data' => $district_wise_data,
                'department_wise_data' => $department_wise_data, 'financial_year_wise_indicator_count' => $financial_year_wise_indicator_count,
                'financial_year_wise_target_count' => $financial_year_wise_target_count, 'district_wise_indicator_count' => $district_wise_indicator_count,
                'district_wise_target_count' => $district_wise_target_count, 'department_wise_indicator_count' => $department_wise_indicator_count,
                'department_wise_target_count' => $department_wise_target_count, 'indicator_type' => $indicator_type, 'error' => $error, 'filter_financial_year' => $filter_financial_year,
                'filter_goal' => $filter_goal, 'filter_district' => $filter_district, 'filter_department' => $filter_department, 'default_year' => $default_year
            ]);
        } else {
            if (isset($request->export) && !empty($request->export) && $request->export == 'excel') {
                $response['status'] = 'error';
                $response['message'] = $error;
                return response()->json($response);
            } else {
                $financial_year_wise_data = array();
                $district_wise_data = array();
                $department_wise_data = array();
                $financial_year_wise_indicator_count = array();
                $financial_year_wise_target_count = array();
                $district_wise_indicator_count = array();
                $district_wise_target_count = array();
                $department_wise_indicator_count = array();
                $department_wise_target_count = array();
                return view('admin_pages.export-report.index', [
                    'districts' => $districts, 'departments' => $departments, 'financialYears' => $financialYear1,
                    'goals' => $goals, 'user' => $user, 'financial_year_wise_data' => $financial_year_wise_data, 'district_wise_data' => $district_wise_data,
                    'department_wise_data' => $department_wise_data, 'financial_year_wise_indicator_count' => $financial_year_wise_indicator_count,
                    'financial_year_wise_target_count' => $financial_year_wise_target_count, 'district_wise_indicator_count' => $district_wise_indicator_count,
                    'district_wise_target_count' => $district_wise_target_count, 'department_wise_indicator_count' => $department_wise_indicator_count,
                    'department_wise_target_count' => $department_wise_target_count, 'indicator_type' => $indicator_type, 'error' => $error,
                    'filter_financial_year' => $filter_financial_year, 'filter_goal' => $filter_goal, 'filter_district' => $filter_district, 'filter_department' => $filter_department
                ]);
            }
        }
    }
}
