<?php

namespace App\Http\Controllers;

use App\Models\District;
use Illuminate\Http\Request;
use Maatwebsite\Excel\Facades\Excel;
use App\Exports\DistrictExport;
use Illuminate\Database\QueryException;
use Illuminate\Support\Facades\Log;
use App\Http\Requests\DistrictRequest;
use App\Http\Requests\UpdateDistrictRequest;
use App\Helpers\CommonHelper;


class DistrictController extends Controller
{

    public function __construct()
    {
        $this->middleware('permission:View District', ['only' => ['index']]);
        $this->middleware('permission:Create District', ['only' => ['create', 'store']]);
        $this->middleware('permission:Update District', ['only' => ['update', 'edit']]);
        $this->middleware('permission:Delete District', ['only' => ['destroy']]);
        $this->middleware('permission:Export District', ['only' => ['exportDistricts']]);
        $this->middleware('auth');
    }

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = District::query();
        try {
            $districts = $query->paginate(env('DISTRICT_PAGINATION_LIMIT',10));
        } catch (QueryException $e) {
            $logData['file'] = __FILE__;
            $logData['method'] = __FUNCTION__;
            $logData['query'] = $e->getSql();
            $logData['exception'] = $e->getMessage();
            // Log exception to file with file name and method information
            CommonHelper::dataActionLogger($logData, 'exception');
            return redirect('districts')->with('error', 'Error while fetching Districts. Please try after sometime.');
        }

        return view('admin_pages.district.index', ['districts' => $districts]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $district = [];
        return view('admin_pages.district.create', ['district' => $district]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(DistrictRequest $request)
    {
        try {
            $district = new District();
            $district->district_name = trim($request->district_name);
            $district->district_code = trim($request->district_code);
            $district->region = trim($request->division);
            $district->save();
        } catch (QueryException $e) {
            $logData['file'] = __FILE__;
            $logData['method'] = __FUNCTION__;
            $logData['query'] = $e->getSql();
            $logData['exception'] = $e->getMessage();
            // Log exception to file with file name and method information
            CommonHelper::dataActionLogger($logData, 'exception');
            return redirect('districts')->with('error', 'Error while creating District. Please try after sometime.');
        }
        return redirect('districts')->with('success', 'District created successfully.');
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $query  = District::query();
        try {
            $district = $query->find($id);
        } catch (QueryException $e) {
            $logData['file'] = __FILE__;
            $logData['method'] = __FUNCTION__;
            $logData['query'] = $e->getSql();
            $logData['exception'] = $e->getMessage();
            // Log exception to file with file name and method information
            CommonHelper::dataActionLogger($logData, 'exception');
            return redirect('districts')->with('error', 'Error while creating District. Please try after sometime.');
        }
        return view('admin_pages.district.create', ['district' => $district]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(UpdateDistrictRequest $request, District $district)
    {
        if (!empty($district->id)) {
            try {
                $district->district_name = trim($request->district_name);
                $district->district_code = trim($request->district_code);
                $district->region = trim($request->division);
                $district->save();
            } catch (QueryException $e) {
                $logData['file'] = __FILE__;
                $logData['method'] = __FUNCTION__;
                $logData['query'] = $e->getSql();
                $logData['exception'] = $e->getMessage();
                // Log exception to file with file name and method information
                CommonHelper::dataActionLogger($logData, 'exception');
                return redirect('districts')->with('error', 'Error while updating District. Please try after sometime.');
            }
        } else {
            return redirect('districts')->with('error', 'District not found.');
        }

        return redirect("districts")->with('success', 'District updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id, string $action)
    {
        $query  = District::query();
        try {
            $district = $query->find($id);
            if (isset($district) && !empty($district)) {
                if (strtolower(trim($action)) == 'inactive') {
                    $district->status = 'inactive';
                    $district->save();
                    return redirect('districts')->with('success', 'District deactivated Successfully.');
                } elseif (strtolower(trim($action)) == 'active') {
                    $district->status = 'active';
                    $district->save();
                    return redirect('districts')->with('success', 'District activated Successfully.');
                } else {
                    return redirect('districts')->with('error', 'Invalid action.');
                }
            } else {
                return redirect('districts')->with('error', 'District not found.');
            }
        } catch (QueryException $e) {
            $logData['file'] = __FILE__;
            $logData['method'] = __FUNCTION__;
            $logData['query'] = $e->getSql();
            $logData['exception'] = $e->getMessage();
            // Log exception to file with file name and method information
            CommonHelper::dataActionLogger($logData, 'exception');
            return redirect('district')->with('error', 'Error while updating the status of District. Please try after sometime.');
        }
    }
    public function show($id)
    {
        // Logic for showing a specific district
    }

    public function exportDistricts()
    {
        $query = District::query();
        try {
            $districts = $query->select('id', 'district_name', 'district_code', 'region')->get();
        } catch (QueryException $e) {
            $logData['file'] = __FILE__;
            $logData['method'] = __FUNCTION__;
            $logData['query'] = $e->getSql();
            $logData['exception'] = $e->getMessage();
            // Log exception to file with file name and method information
            CommonHelper::dataActionLogger($logData, 'exception');
            return redirect('districts')->with('error', 'Error while exporting Districts. Please try after sometime.');
        }

        if ($districts->isEmpty()) {
            return redirect('districts')->with('error', 'No Districts found to export.');
        }

        $filename = 'list_of_districts_' . now()->format('Y-m-d_H-i-s') . '.xlsx';
        Excel::store(new DistrictExport($districts), 'export/districts/' . $filename);

        // Generate the full path to the saved file
        $filePath = storage_path('app/export/districts/' . $filename);

        return response()->download($filePath, $filename);
    }
}
