<?php

namespace App\Http\Controllers;

use App\Models\Department;
use Illuminate\Http\Request;
use Maatwebsite\Excel\Facades\Excel;
use App\Exports\DepartmentExport;
use Illuminate\Database\QueryException;
use Illuminate\Support\Facades\Log;
use App\Helpers\CommonHelper;

class DepartmentController extends Controller
{
    public function __construct()
    {
        $this->middleware('permission:View Department', ['only' => ['index']]);
        $this->middleware('permission:Create Department', ['only' => ['create', 'store']]);
        $this->middleware('permission:Update Department', ['only' => ['update', 'edit']]);
        $this->middleware('permission:Delete Department', ['only' => ['destroy']]);
        $this->middleware('permission:Export Department', ['only' => ['exportDepartments']]);
        $this->middleware('auth');
    }

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = Department::query();
        if ($request->has('search')) {
            $query->where('id', $request->search);
        }
        try {
            $departments = $query->paginate(env('DISTRICT_PAGINATION_LIMIT', 10));
        } catch (QueryException $e) {
            $logData['file'] = __FILE__;
            $logData['method'] = __FUNCTION__;
            $logData['query'] = $e->getSql();
            $logData['exception'] = $e->getMessage();
            // Log exception to file with file name and method information
            CommonHelper::dataActionLogger($logData, 'exception');
            return redirect('departments')->with('error', 'Error while fetching Departments. Please try after sometime.');
        }
        return view('admin_pages.department.index', ['departments' => $departments]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $department = [];
        return view('admin_pages.department.create', ['departments' => $department]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'department_name' => 'required|string|min:3|max:191',
        ]);

        try {
            $department = new Department();
            $department->department_name = trim($request->department_name);
            $department->date_added = now();
            $department->save();
        } catch (QueryException $e) {
            $logData['file'] = __FILE__;
            $logData['method'] = __FUNCTION__;
            $logData['query'] = $e->getSql();
            $logData['exception'] = $e->getMessage();
            // Log exception to file with file name and method information
            CommonHelper::dataActionLogger($logData, 'exception');
            return redirect('departments')->with('error', 'Error while creating Department. Please try after sometime.');
        }
        return redirect('departments')->with('success', 'Department created successfully.');
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $query  = Department::query();
        try {
            $department = $query->find($id);
        } catch (QueryException $e) {
            $logData['file'] = __FILE__;
            $logData['method'] = __FUNCTION__;
            $logData['query'] = $e->getSql();
            $logData['exception'] = $e->getMessage();
            // Log exception to file with file name and method information
            CommonHelper::dataActionLogger($logData, 'exception');
            return redirect('departments')->with('error', 'Error while creating Department. Please try after sometime.');
        }
        return view('admin_pages.department.create', ['departments' => $department]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Department $department)
    {
        $request->validate([
            'department_name' => 'required|string',
        ]);

        if (!empty($department->id)) {
            try {
                $department->department_name = trim($request->department_name);
                $department->save();
            } catch (QueryException $e) {
                $logData['file'] = __FILE__;
                $logData['method'] = __FUNCTION__;
                $logData['query'] = $e->getSql();
                $logData['exception'] = $e->getMessage();
                // Log exception to file with file name and method information
                CommonHelper::dataActionLogger($logData, 'exception');
                return redirect('departments')->with('error', 'Error while updating Department. Please try after sometime.');
            }
        } else {
            return redirect('departments')->with('error', 'Department not found.');
        }

        return redirect("departments")->with('success', 'Department updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id, string $action)
    {
        $query  = Department::query();
        try {
            $department = $query->find($id);
            if (isset($department) && !empty($department)) {
                if ($action == 'active') {
                    $department->status = 'Active';
                    $department->save();
                    return redirect('departments')->with('success', 'Department activated successfully.');
                } else if ($action == 'inactive') {
                    $department->status = 'Inactive';
                    $department->save();
                    return redirect('departments')->with('success', 'Department deactivated successfully.');
                } else {
                    return redirect('departments')->with('error', 'Invalid action.');
                }
            } else {
                return redirect('departments')->with('error', 'Department not found.');
            }
        } catch (QueryException $e) {
            $logData['file'] = __FILE__;
            $logData['method'] = __FUNCTION__;
            $logData['query'] = $e->getSql();
            $logData['exception'] = $e->getMessage();
            // Log exception to file with file name and method information
            CommonHelper::dataActionLogger($logData, 'exception');
            return redirect('departments')->with('error', 'Error updating status of Department. Please try after sometime.');
        }
    }

    public function show($id)
    {
        // Logic for showing a specific department
    }

    public function exportDepartments()
    {
        $query = Department::query();
        try {
            $departments = $query->select('id', 'department_name', 'status', 'date_added', 'date_updated')->get();
        } catch (QueryException $e) {
            $logData['file'] = __FILE__;
            $logData['method'] = __FUNCTION__;
            $logData['query'] = $e->getSql();
            $logData['exception'] = $e->getMessage();
            // Log exception to file with file name and method information
            CommonHelper::dataActionLogger($logData, 'exception');
            return redirect('departments')->with('error', 'Error while exporting Departments. Please try after sometime.');
        }

        if ($departments->isEmpty()) {
            return redirect('departments')->with('error', 'No Departments found to export.');
        }

        $filename = 'list_of_departments_' . now()->format('Y-m-d_H-i-s') . '.xlsx';
        Excel::store(new DepartmentExport($departments), 'export/department/' . $filename);

        // Generate the full path to the saved file
        $filePath = storage_path('app/export/department/' . $filename);

        return response()->download($filePath, $filename);
    }
}
