<?php

namespace App\Helpers;

use App\Models\Goal;
use Illuminate\Database\QueryException;
use Illuminate\Support\Facades\Log;
use App\Models\District;
use App\Models\Target;
use App\Models\Department;
use Monolog\Logger;
use Monolog\Handler\StreamHandler;
use Monolog\Formatter\LineFormatter;
use App\Models\ReportingYear;
use App\Models\Indicator;
use Illuminate\Support\Facades\Auth;
use App\Models\User;
use App\Models\Role;
use App\Models\Permission;

class CommonHelper
{
    /* Start:   
    * Function Name: exception_logger
    * Log the exception to the file.
    * This method logs the exception to the file with the file name and method information.
    *
    * @param  array $data_array The array containing the exception, file name, method, and query information.
    * @return bool               The boolean value indicating whether the exception was logged successfully.
    * -------------------------------------------------------------------------
    * AUTHOR: SUHEL KHAN
    * DATE: 25-April-2024
    * PM ID: #148098
    * CHANGES: FIRST WRITE-UP
    * -------------------------------------------------------------------------
    */
    public static function sql_exception_logger($data_array)
    {
        if (isset($data_array['exception']) && !empty($data_array['exception'])) {
            Log::channel('exception_log')->critical('SQL query exception', [
                'exception' => $data_array['exception'],
                'file' => $data_array['file'],
                'method' => $data_array['method'],
                'query' => $data_array['query'],
            ]);
            return true;
        }
        return false;
    }
    /*End: exception_logger*/

    /**START
     * Function - getGoalData
     * Get data for one or all goals.
     *
     * This method retrieves data for one or all goals from the database.
     * If 'all' is passed as the parameter, it returns data for all goals.
     * Otherwise, it returns data for the specified goal ID.
     *
     * @param  string|int|array $goalIds (Optional) The ID(s) of the goal(s) to retrieve data for.
     *                                    If 'all' is passed, data for all goals will be retrieved.
     * @return array                      An array containing data for the specified goal(s).
     * 
     * --------------------------------------------------------------------------------------------
     * AUTHOR: SUHEL KHAN
     * DATE: 19-April-2024
     * PM ID: #148098
     * CHANGES: FIRST WRITE-UP
     * --------------------------------------------------------------------------------------------
     */
    public static function getGoalData($goalIds = 'all')
    {
        $goalData = [];
        if ($goalIds == 'all') {
            try {
                $goals = Goal::all();
            } catch (QueryException $e) {
                $logData['file'] = __FILE__;
                $logData['method'] = __FUNCTION__;
                $logData['query'] = $e->getSql();
                $logData['exception'] = $e->getMessage();
                // Log exception to file with file name and method information
                CommonHelper::dataActionLogger($logData, 'exception');
                return false;
            }
            if ($goals->isEmpty()) {
                return $goalData;
            }
            foreach ($goals as $goal) {
                $goalData[] = [
                    'goal_id' => $goal->id,
                    'goal_name' => $goal->goal_name,
                    'image_name' => 'goal_' . $goal->id // Assuming image name format is 'goal_{goal_id}'
                ];
            }
        } else {
            try {
                $goals = Goal::where('id', $goalIds)->first();
            } catch (QueryException $e) {
                $logData['file'] = __FILE__;
                $logData['method'] = __FUNCTION__;
                $logData['query'] = $e->getSql();
                $logData['exception'] = $e->getMessage();
                // Log exception to file with file name and method information
                CommonHelper::dataActionLogger($logData, 'exception');
                return false;
            }
            if (!isset($goals) && empty($goals)) {
                return $goalData;
            }

            $goalData = [
                'goal_id' => $goals->id,
                'goal_name' => $goals->goal_name,
                'image_name' => 'goal_' . $goals->id // Assuming image name format is 'goal_{goal_id}'
            ];
        }
        return $goalData;
    }
    /**END -getGoalData*/

    /**Start
     * Function getDistrictApprovers
     * Get all the district approvers for the particular district.
     * 
     * --------------------------------------------------------------------------------------------
     * AUTHOR: Almaaz Ahmed
     * DATE: 07-May-2025
     * PM ID: #159987
     * CHANGES: FIRST WRITE-UP
     * --------------------------------------------------------------------------------------------
     */
    public static function getDistrictApprovers($districtId)
    {
        try {
            $user = User::where('district_id', $districtId)->get();
            $approvers = $user->filter(function ($user) {
                return $user->hasRole('District-Approver');
            })->values(); // reset keys

            return $approvers;
        } catch (QueryException $e) {
            $logData['file'] = __FILE__;
            $logData['method'] = __FUNCTION__;
            $logData['query'] = $e->getSql();
            $logData['exception'] = $e->getMessage();
            CommonHelper::dataActionLogger($logData, 'exception');
            return collect(); // return empty collection instead of false
        }
    }
    /**End -getDistrictApprovers*/

    /**Start
     * Function: getDepartmentApprovers
     * Get all the department approvers for the particular department.
     * 
     * --------------------------------------------------------------------------------------------
     * AUTHOR: Almaaz Ahmed
     * DATE: 07-May-2025
     * PM ID: #159987
     * CHANGES: FIRST WRITE-UP
     * --------------------------------------------------------------------------------------------
     */
    public static function getDepartmentApprovers($departmentId)
    {
        try {
            $users = User::where('department_id', $departmentId)->get();
            $approvers = $users->filter(function ($user) {
                return $user->hasRole('Department-Approver');
            })->values(); // reset keys

            return $approvers;
        } catch (QueryException $e) {
            $logData['file'] = __FILE__;
            $logData['method'] = __FUNCTION__;
            $logData['query'] = $e->getSql();
            $logData['exception'] = $e->getMessage();
            CommonHelper::dataActionLogger($logData, 'exception');
            return collect(); // return empty collection instead of false
        }
    }
    /**End getDepartmentApprovers */

    /**
     * Function: getDistrictHodApprover
     * Get all the district HOD approvers for the particular district.
     * 
     * --------------------------------------------------------------------------------------------
     * AUTHOR: Almaaz Ahmed
     * DATE: 07-May-2025
     * PM ID: #159987
     * CHANGES: FIRST WRITE-UP
     * --------------------------------------------------------------------------------------------
     */
    public static function getDistrictHodApprover($districtId){
        try {
            $user = User::where('district_id', $districtId)->get();
            $approvers = $user->filter(function ($user) {
                return $user->hasRole('District-HOD');
            })->values(); // reset keys

            return $approvers;
        } catch (QueryException $e) {
            $logData['file'] = __FILE__;
            $logData['method'] = __FUNCTION__;
            $logData['query'] = $e->getSql();
            $logData['exception'] = $e->getMessage();
            CommonHelper::dataActionLogger($logData, 'exception');
            return collect(); // return empty collection instead of false
        }
    }
    /**End getDistrictHodApprover */

    /**
     * Function: getDepartmentHodApprover
     * Get all the department HOD approvers for the particular department.
     * 
     * --------------------------------------------------------------------------------------------
     * AUTHOR: Almaaz Ahmed
     * DATE: 07-May-2025
     * PM ID: #159987
     * CHANGES: FIRST WRITE-UP
     * --------------------------------------------------------------------------------------------
     */
    public static function getDepartmentHodApprover($departmentId){
        try{
            $user = User::where('department_id', $departmentId)->get();
            $approvers = $user->filter(function ($user) {
                return $user->hasRole('Department-HOD');
            })->values(); // reset keys

            return $approvers;
        }catch (QueryException $e) {
            $logData['file'] = __FILE__;
            $logData['method'] = __FUNCTION__;
            $logData['query'] = $e->getSql();
            $logData['exception'] = $e->getMessage();
            CommonHelper::dataActionLogger($logData, 'exception');
            return collect(); // return empty collection instead of false
        }
    }
    /**End getDepartmentHodApprover */

    /**
     * Function: getAdmin
     * Get all the admin users.
     * 
     * --------------------------------------------------------------------------------------------
     * AUTHOR: Almaaz Ahmed
     * DATE: 07-May-2025
     * PM ID: #159987
     * CHANGES: FIRST WRITE-UP
     * --------------------------------------------------------------------------------------------
     */
    public static function getAdmin(){
        try{
            // get all the users
            $users = User::all();
            // filter the users to get the admin
            $admin = $users->filter(function ($user) {
                return $user->hasRole('Admin');
            })->values(); // reset keys

            return $admin;
        }catch (QueryException $e) {
            $logData['file'] = __FILE__;
            $logData['method'] = __FUNCTION__;
            $logData['query'] = $e->getSql();
            $logData['exception'] = $e->getMessage();
            CommonHelper::dataActionLogger($logData, 'exception');
            return collect(); // return empty collection instead of false
        }
    }
    /**End getAdmin */

    /**START
     * Function - getLastThreeYears
     * Get the last three years.
     *
     * This method generates the last three years based on the default year.
     *
     * @param  string $defaultYear The default year to generate the last three years from.
     * @return array               An array containing the last three years.
     * 
     * --------------------------------------------------------------------------------------------
     * AUTHOR: SUHEL KHAN
     * DATE: 19-April-2024
     * PM ID: #148098
     * CHANGES: FIRST WRITE-UP
     * --------------------------------------------------------------------------------------------
     */
    public static function getLastThreeYears($defaultYear)
    {
        $years = [];
        // Extract the year from the default year
        $defaultYear = (int)$defaultYear;
        // Generate the last three years
        for ($i = $defaultYear - 2; $i <= $defaultYear; $i++) {
            $years[] = $i . '-' . ($i + 1);
        }
        return $years;
    }
    /**END -getLastThreeYears*/

    /**START
     * Function - getReportingYears
     * Get the reporting years.
     *
     * This method generates the reporting years based on the last three years.
     *
     * @param  array $lastThreeYears An array containing the last three years.
     * @return array                 An array containing the reporting years.
     * 
     * --------------------------------------------------------------------------------------------
     * AUTHOR: SUHEL KHAN
     * DATE: 19-April-2024
     * PM ID: #148098
     * CHANGES: FIRST WRITE-UP
     * --------------------------------------------------------------------------------------------
     */
    public static function getReportingYears($lastThreeYears)
    {
        $years = [];
        foreach ($lastThreeYears as $data) {
            $year = explode('-', $data);
            $years[] = [
                'id' => $data,
                'year' => $year[0],
            ];
        }
        return $years;
    }
    /**END -getReportingYears*/

    /**START
     * Function - getDistricts
     * Get data for one or all districts.
     *
     * This method retrieves data for one or all districts from the database.
     * If 'all' is passed as the parameter, it returns data for all districts.
     * If 'default' is passed as the parameter, it returns data for the default district.
     * Otherwise, it returns data for the specified district ID.
     *
     * @param  string|int $request (Optional) The ID of the district to retrieve data for.
     *                             If 'all' is passed, data for all districts will be retrieved.
     *                             If 'default' is passed, data for the default district will be retrieved.
     * @return array               An array containing data for the specified district(s).
     * 
     * --------------------------------------------------------------------------------------------
     * AUTHOR: SUHEL KHAN
     * DATE: 08-May-2024
     * PM ID: #148098
     * CHANGES: FIRST WRITE-UP
     * --------------------------------------------------------------------------------------------
     */
    public static function getDistricts($request = 'all')
    {
        $districts = [];
        $query = District::query();
        if ($request == 'all') {
            try {
                $query->where('status', 'Active');
                $districts = $query->orderBy('rank', 'asc')->orderBy('district_name', 'asc')->get();
            } catch (QueryException $e) {
                $logData['file'] = __FILE__;
                $logData['method'] = __FUNCTION__;
                $logData['query'] = $e->getSql();
                $logData['exception'] = $e->getMessage();
                // Log exception to file with file name and method information
                CommonHelper::dataActionLogger($logData, 'exception');
                return false;
            }
            if ($districts->isEmpty()) {
                return $districts;
            }
        } else if ($request == 'default') {
            try {
                $districts = District::select('id', 'district_name')
                    ->where('is_default_district', '1')
                    ->where('status', 'Active')
                    ->first();
            } catch (QueryException $e) {
                $logData['file'] = __FILE__;
                $logData['method'] = __FUNCTION__;
                $logData['query'] = $e->getSql();
                $logData['exception'] = $e->getMessage();
                // Log exception to file with file name and method information
                CommonHelper::dataActionLogger($logData, 'exception');
                return false;
            }
            if (empty($districts)) {
                return $districts;
            }
        } else {
            try {
                $districts = District::where('id', $request)->first();
            } catch (QueryException $e) {
                $logData['file'] = __FILE__;
                $logData['method'] = __FUNCTION__;
                $logData['query'] = $e->getSql();
                $logData['exception'] = $e->getMessage();
                // Log exception to file with file name and method information
                CommonHelper::dataActionLogger($logData, 'exception');
                return false;
            }
            if (!isset($districts) && empty($districts)) {
                return $districts;
            }
        }
        return $districts;
    }
    /**END -getDistricts*/

    /**START
     * Function - getTargets
     * Get data for one or all targets.
     *
     * This method retrieves data for one or all targets from the database.
     * If 'all' is passed as the parameter, it returns data for all targets.
     * Otherwise, it returns data for the specified target ID.
     *
     * @param  string|int $request (Optional) The ID of the target to retrieve data for.
     *                             If 'all' is passed, data for all targets will be retrieved.
     * @param  int $goal_id       (Optional) The ID of the goal to retrieve data for.
     * @return array               An array containing data for the specified target(s).
     * 
     * --------------------------------------------------------------------------------------------
     * AUTHOR: SUHEL KHAN
     * DATE: 08-May-2024
     * PM ID: #148098
     * CHANGES: FIRST WRITE-UP
     * --------------------------------------------------------------------------------------------
     */
    public static function getTargets($request = 'all', $goal_id = null)
    {
        $targets = [];
        $query = Target::query();
        $query->select('target_id', 'target_name', 'target_number');
        $query->where('status', 'Active');
        //to fetch all targets and there is no goal_id passed in the parameter
        if ($request == 'all' && $goal_id == null) {
            try {
                $targets = $query->get();
            } catch (QueryException $e) {
                $logData['file'] = __FILE__;
                $logData['method'] = __FUNCTION__;
                $logData['query'] = $e->getSql();
                $logData['exception'] = $e->getMessage();
                // Log exception to file with file name and method information
                CommonHelper::dataActionLogger($logData, 'exception');
                return false;
            }
            if ($targets->isEmpty()) {
                return $targets;
            }
            //to fetch all targets and there is a goal_id passed in the parameter
        } else if ($request == 'all' && $goal_id != null) {
            try {
                $targets = $query->where('goal_id', $goal_id)->get();
            } catch (QueryException $e) {
                $logData['file'] = __FILE__;
                $logData['method'] = __FUNCTION__;
                $logData['query'] = $e->getSql();
                $logData['exception'] = $e->getMessage();
                // Log exception to file with file name and method information
                CommonHelper::dataActionLogger($logData, 'exception');
                return false;
            }
            if ($targets->isEmpty()) {
                return $targets;
            }
            //to fetch a specific target and there is a goal_id passed in the parameter
        } else if ($request != "all" && $goal_id != null) {
            try {
                $query->where('target_id', $request);
                $targets = $query->where('goal_id', $goal_id)->get();
            } catch (QueryException $e) {
                $logData['file'] = __FILE__;
                $logData['method'] = __FUNCTION__;
                $logData['query'] = $e->getSql();
                $logData['exception'] = $e->getMessage();
                // Log exception to file with file name and method information
                CommonHelper::dataActionLogger($logData, 'exception');
                return false;
            }
            if (!isset($targets) && empty($targets)) {
                return $targets;
            }
            //to fetch a specific target and there is no goal_id passed in the parameter
        } else {
            try {
                $targets = $query->where('target_id', $request)->first();
            } catch (QueryException $e) {
                $logData['file'] = __FILE__;
                $logData['method'] = __FUNCTION__;
                $logData['query'] = $e->getSql();
                $logData['exception'] = $e->getMessage();
                // Log exception to file with file name and method information
                CommonHelper::dataActionLogger($logData, 'exception');
                return false;
            }
            if (!isset($targets) && empty($targets)) {
                return $targets;
            }
        }
        return $targets;
    }
    /**END -getTargets*/

    /**START
     * Function - getDepartments
     * Get data for one or all departments.
     *
     * This method retrieves data for one or all departments from the database.
     * If 'all' is passed as the parameter, it returns data for all departments.
     * Otherwise, it returns data for the specified department ID.
     *
     * @param  string|int $request (Optional) The ID of the department to retrieve data for.
     *                             If 'all' is passed, data for all departments will be retrieved.
     * @return array               An array containing data for the specified department(s).
     * 
     * --------------------------------------------------------------------------------------------
     * AUTHOR: SUHEL KHAN
     * DATE: 08-May-2024
     * PM ID: #148098
     * CHANGES: FIRST WRITE-UP
     * --------------------------------------------------------------------------------------------
     */
    public static function getDepartments($request = 'all')
    {
        $departments = [];
        $query = Department::query();
        if ($request == 'all') {
            try {
                $departments = $query->where('status', 'Active')->get();
            } catch (QueryException $e) {
                $logData['file'] = __FILE__;
                $logData['method'] = __FUNCTION__;
                $logData['query'] = $e->getSql();
                $logData['exception'] = $e->getMessage();
                // Log exception to file with file name and method information
                CommonHelper::dataActionLogger($logData, 'exception');
                return false;
            }
            if ($departments->isEmpty()) {
                return $departments;
            }
        } else if ($request == 'default') {
            try {
                $departments = $query->where('is_default', 'Yes')->where('status', 'Active')->first();
            } catch (QueryException $e) {
                $logData['file'] = __FILE__;
                $logData['method'] = __FUNCTION__;
                $logData['query'] = $e->getSql();
                $logData['exception'] = $e->getMessage();
                // Log exception to file with file name and method information
                CommonHelper::dataActionLogger($logData, 'exception');
                return false;
            }
            if (empty($departments)) {
                return $departments;
            }
        } else {
            try {
                $departments = $query->where('id', $request)->first();
            } catch (QueryException $e) {
                $logData['file'] = __FILE__;
                $logData['method'] = __FUNCTION__;
                $logData['query'] = $e->getSql();
                $logData['exception'] = $e->getMessage();
                // Log exception to file with file name and method information
                CommonHelper::dataActionLogger($logData, 'exception');
                return false;
            }
            if (!isset($departments) && empty($departments)) {
                return $departments;
            }
        }
        return $departments;
    }
    /**END -getDepartments*/


    public static function getIndicators($request = 'all', $goal_id = null, $target_id = null, $type = null, $notAbsolute = '')
    {
        $indicators = [];
        $query = Indicator::query();
        $query->select('id', 'indicator_name', 'indicator_number');
        $query->where('status', 'Active');

        if ($request != "all") {
            $query->where('id', $request);
        }

        if ($goal_id != null) {
            $query->where('goal_id', $goal_id);
        }

        if ($target_id != null) {
            $query->where('target_id', $target_id);
        }

        if ($type != null) {
            if ($type == 'utif') {
                $query->where('is_utif', 'Yes');
            } else if ($type == 'dif') {
                $query->where('is_dif', 'Yes');
            }
        }
        if($notAbsolute != '' && $notAbsolute == 'yes'){
            $query->whereNot('unit', 'Absolute');

        }

        try {
            $indicators = $query->get();
        } catch (QueryException $e) {
            $logData['file'] = __FILE__;
            $logData['method'] = __FUNCTION__;
            $logData['query'] = $e->getSql();
            $logData['exception'] = $e->getMessage();
            // Log exception to file with file name and method information
            CommonHelper::dataActionLogger($logData, 'exception');
            return false;
        }
        if ($indicators->isEmpty()) {
            return $indicators;
        }
        return $indicators;
    }

    public static function getReoprtingYears()
    {
        // Get the default year as an integer
        $defaultYear = ReportingYear::where('is_default', 'yes')->limit(1)->value('reporting_year');

        // Fetch years up to the default year, sorted as integers
        $years = ReportingYear::where('status', 'active')
            ->whereRaw('CAST(reporting_year AS UNSIGNED) <= ?', [$defaultYear])
            ->orderByRaw('CAST(reporting_year AS UNSIGNED)')
            ->pluck('reporting_year')
            ->sort();

        try {
            $singleYears = $years->flatMap(function ($year) {
                return explode('-', $year);
            })->map(function ($year) {
                return (int) $year;
            })->unique()->values()->sort();
        } catch (QueryException $e) {
            $logData['file'] = __FILE__;
            $logData['method'] = __FUNCTION__;
            $logData['query'] = $e->getSql();
            $logData['exception'] = $e->getMessage();
            // Log exception to file with file name and method information
            CommonHelper::dataActionLogger($logData, 'exception');
            return false;
        }

        /*
        * -------------------------------------------------------------------------
            BOC
            Task #156928 The representation of the year should be the financial year i.e. 2021-2022 throughout the application. 
            Updated the below code so to generate the pair years in the format '2011-2013'
            @author Almaaz Ahmed
            @date 18-12-2024 
        * -------------------------------------------------------------------------    
        */
        $pairYears = collect();
        for ($i = 0; $i <= $singleYears->count(); $i++) {
            if (isset($singleYears[$i + 2])) {
                $pairYears->push($singleYears[$i] . '-' . $singleYears[$i + 2]);
            }
        }
        /*
        * -------------------------------------------------------------------------
            BOC
            Task #156928 The representation of the year should be the financial year i.e. 2021-2022 throughout the application. 
            Removed the below code as it is not required anymore as we are using financial year
            @author Almaaz Ahmed
            @date 18-12-2024 
        * -------------------------------------------------------------------------    
        */
        return ['singleYears' => $singleYears->values(), 'pairYears' => $pairYears->values(), 'district_indicator_pair' => $years];
    }

    public static function getDefaultYear()
    {
        try {
            $year = ReportingYear::where('is_default', 'yes')->value('reporting_year');
        } catch (QueryException $e) {
            $logData['file'] = __FILE__;
            $logData['method'] = __FUNCTION__;
            $logData['query'] = $e->getSql();
            $logData['exception'] = $e->getMessage();
            // Log exception to file with file name and method information
            CommonHelper::dataActionLogger($logData, 'exception');
            return false;
        }
        $years = explode('-', $year);
        $year = $years[0];
        return $year;
    }

    /**START
     * Function - dataActionLogger
     * Log all types of data using only one function (Exception, Error and Information).
     * This method logs the information to the file with the file name, method, user, and information.
     *
     * @param  string $file     The name of the file to log the information to.
     * @param  string $location The location of the file to log the information to.
     * @param  array  $data     The array containing the file, method, user, and information.
     * @return void
     * -------------------------------------------------------------------------
     * AUTHOR: SUHEL KHAN
     * DATE: 25-April-2024
     * PM ID: #148098
     * CHANGES: FIRST WRITE-UP
     * -------------------------------------------------------------------------
     */
    public static function dataActionLogger($logData, $log_type, $location = '', $file = '')
    {
        // BOC
        // Added changes to send the User Name as 'Public User' in case log entry needs to be added on the Public Dashboard or on the page which are accessible without User Authentication.
        // Auther : Almaaz Ahmed
        // Date : 10-January-2025
        if(Auth::check()){
            $user = Auth::user()->email;
        }else{
            $user = 'Public User';
        }
        // EOC
        if (!empty($logData) && count($logData) > 0) {

            if ($log_type == 'exception' && !empty($logData['exception'])) {
                $file = 'exceptions_' . date('Y-m-d') . '.txt';
                $path = storage_path('logs/exception-log/' . $file);

                $logger = new Logger('sql_exception_log');

                $handler = new StreamHandler($path, Logger::DEBUG);

                $formatter = new LineFormatter('DateTime: [%datetime%] | File: %context.file% | Method: %context.method% | User: %context.user% | Query: %context.query% | Exception: %context.exception%' . PHP_EOL);
                $handler->setFormatter($formatter);

                // Push the handler to the logger
                $logger->pushHandler($handler);

                // Log the exception
                $logger->Critical('', [
                    'file' => $logData['file'],
                    'method' => $logData['method'],
                    'user' => $user,
                    'query' => $logData['query'],
                    'exception' => $logData['exception']
                ]);
            } else if ($log_type == 'error' && !empty($logData['error'])) {
                $file = 'error_' . date('Y-m-d') . '.txt';
                $path = storage_path('logs/error-log/' . $file);

                $logger = new Logger('error_log');

                $handler = new StreamHandler($path, Logger::DEBUG);

                $formatter = new LineFormatter('DateTime: [%datetime%] | File: %context.file% | Method: %context.method% | User: %context.user% | Error: %context.error%' . PHP_EOL);
                $handler->setFormatter($formatter);

                // Push the handler to the logger
                $logger->pushHandler($handler);

                // Log the exception
                $logger->Error('', [
                    'file' => $logData['file'],
                    'method' => $logData['method'],
                    'user' => $user,
                    'error' => $logData['error']
                ]);
            } else if ($log_type == 'info' && !empty($logData['information'])) {
                $path = storage_path($location . '/' . $file);
                // Create a new logger instance
                $logger = new Logger('info_log');

                // Create a handler with a dynamic file path
                $handler = new StreamHandler($path, Logger::DEBUG);

                // Set the formatter
                $formatter = new LineFormatter('DateTime: [%datetime%] File: %context.file% | Method: %context.method% | User: %context.user% | Information: %context.information%' . PHP_EOL);
                $handler->setFormatter($formatter);

                // Push the handler to the logger
                $logger->pushHandler($handler);

                // Log the exception
                $logger->info('', [
                    'file' => $logData['file'],
                    'method' => $logData['method'],
                    'user' => $user,
                    'information' => $logData['information']
                ]);
            }
        }
    }
}
