<?php

namespace App\Console;

use Illuminate\Console\Scheduling\Schedule;
use Illuminate\Foundation\Console\Kernel as ConsoleKernel;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use App\Helpers\CommonHelper;
use App\Jobs\ProcessEntriesJob;
use Illuminate\Support\Facades\Log;

class Kernel extends ConsoleKernel
{
    /**
     * Define the application's command schedule.
     */
    protected function schedule(Schedule $schedule)
    {
        $schedule->call(function () {
            Log::info('Scheduled task started.');

            $year = CommonHelper::getDefaultYear();
            if (!empty($year)) {
                $default_year = $year . '-' . ($year + 1);
            } else {
                $default_year = '';
            }
            $totalIndicators = DB::table('indicator_data_entries')
            ->join('indicators', 'indicators.id', '=', 'indicator_data_entries.indicator_id')
            ->whereNotNull('indicator_data_entries.district_id')
            ->where('indicators.unit', '!=', 'Absolute')
            ->where('indicator_data_entries.status', 'Approved')
            ->where('indicator_data_entries.reporting_year', $default_year)
                ->count();

            $chunkSize = 500;
            $batchId = 'ProcessEntriesJobBatch_' . now()->timestamp;
            $totalChunks = ceil($totalIndicators / $chunkSize);
            Cache::put($batchId . '_total', $totalChunks, now()->addDay());
            Cache::put($batchId . '_completed', 0, now()->addDay());

            DB::table('indicator_data_entries')
            ->join('indicators', 'indicators.id', '=', 'indicator_data_entries.indicator_id')
            ->select('indicator_data_entries.*', 'indicators.is_positive_indicator')
            ->whereNotNull('indicator_data_entries.district_id')
            ->where('indicators.unit', '!=', 'Absolute')
            ->where('indicator_data_entries.reporting_year', $default_year)
                ->where('indicator_data_entries.status', 'Approved')
                ->chunkById($chunkSize, function ($entries) use ($batchId) {
                    Log::info('Dispatching job for batch ID: ' . $batchId);
                    ProcessEntriesJob::dispatch($entries, $batchId);
                });

            Log::info('Scheduled task completed.');
        })->everyMinute();
    }


    /**
     * Register the commands for the application.
     */
    protected function commands(): void
    {
        $this->load(__DIR__.'/Commands');

        require base_path('routes/console.php');
    }
}
